<?php
/**
 * Plugin Name: Otter Fixer Task Board
 * Plugin URI:  https://otterfixer.com
 * Description: A task board for the WordPress admin area with due dates, recurring tasks, templates, bulk actions, activity logs, and CSV export.
 * Version: 1.2.9
 * Author:      Otter Fixer
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: otterfixer-task-board
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

final class OtterFixer_Task_Board {

    const OPT_MODE           = 'of_tb_mode';            // per_user|shared
    const OPT_ASSIGN         = 'of_tb_allow_assign';    // yes|no
    const OPT_NOTICES        = 'of_tb_enable_notices';  // yes|no
    const OPT_CATEGORIES     = 'of_tb_categories';      // array slug => [label,color]
    const OPT_WIDGET_LIMIT   = 'of_tb_widget_limit';    // int
    const OPT_DELETE_DATA    = 'of_tb_delete_data_on_uninstall'; 
    const OPT_DATE_FORMAT   = 'of_tb_date_format';
// yes|no

    const CAP_VIEW   = 'read';
    const CAP_MANAGE = 'manage_options';

    const TRANSIENT_TEMPLATE_DRAFT = 'of_tb_tpl_draft_';

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );

        add_action( 'wp_dashboard_setup', array( $this, 'register_dashboard_widget' ) );
        add_action( 'admin_notices', array( $this, 'overdue_notice' ) );

        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );

        add_action( 'admin_post_of_tb_add', array( $this, 'handle_add' ) );
        add_action( 'admin_post_of_tb_toggle', array( $this, 'handle_toggle' ) );
        add_action( 'admin_post_of_tb_delete', array( $this, 'handle_delete' ) );
        add_action( 'admin_post_of_tb_bulk', array( $this, 'handle_bulk' ) );
        add_action( 'admin_post_of_tb_clear_completed', array( $this, 'handle_clear_completed' ) );

        add_action( 'admin_post_of_tb_add_template', array( $this, 'handle_add_builtin_template' ) );
        add_action( 'admin_post_of_tb_prepare_template_from_tasks', array( $this, 'handle_prepare_template_from_tasks' ) );
        add_action( 'admin_post_of_tb_create_user_template', array( $this, 'handle_create_user_template' ) );
        add_action( 'admin_post_of_tb_apply_user_template', array( $this, 'handle_apply_user_template' ) );
        add_action( 'admin_post_of_tb_delete_user_template', array( $this, 'handle_delete_user_template' ) );
        add_action( 'admin_post_of_tb_export_templates', array( $this, 'handle_export_templates_csv' ) );
        add_action( 'admin_post_of_tb_import_templates', array( $this, 'handle_import_templates_csv' ) );
        add_action( 'admin_post_of_tb_download_templates_template', array( $this, 'handle_download_templates_template_csv' ) );

        add_action( 'admin_post_of_tb_export_tasks', array( $this, 'handle_export_tasks_csv' ) );
        add_action( 'admin_post_of_tb_export_log', array( $this, 'handle_export_log_csv' ) );

        add_action( 'wp_ajax_of_tb_reorder', array( $this, 'ajax_reorder' ) );
    }

    /* -----------------------------
     * Admin menu and settings
     * ----------------------------- */

    public function admin_menu() {
        add_menu_page(
            __( 'Task Board', 'otterfixer-task-board' ),
            __( 'Task Board', 'otterfixer-task-board' ),
            self::CAP_VIEW,
            'otterfixer-task-board',
            array( $this, 'render_page' ),
            'dashicons-clipboard',
            58
        );
    }

    public function register_settings() {
        register_setting( 'of_tb_settings', self::OPT_MODE, array(
            'type'              => 'string',
            'sanitize_callback' => array( $this, 'sanitize_mode' ),
            'default'           => 'per_user',
        ) );

        register_setting( 'of_tb_settings', self::OPT_ASSIGN, array(
            'type'              => 'string',
            'sanitize_callback' => array( $this, 'sanitize_yesno' ),
            'default'           => 'yes',
        ) );

        register_setting( 'of_tb_settings', self::OPT_NOTICES, array(
            'type'              => 'string',
            'sanitize_callback' => array( $this, 'sanitize_yesno' ),
            'default'           => 'yes',
        ) );

        register_setting( 'of_tb_settings', self::OPT_WIDGET_LIMIT, array(
            'type'              => 'integer',
            'sanitize_callback' => array( $this, 'sanitize_widget_limit' ),
            'default'           => 5,
        ) );

        register_setting( 'of_tb_settings', self::OPT_CATEGORIES, array(
            'type'              => 'array',
            'sanitize_callback' => array( $this, 'sanitize_categories' ),
            'default'           => $this->default_categories(),
        ) );

        register_setting( 'of_tb_settings', self::OPT_DELETE_DATA, array(
            'type'              => 'string',
            'sanitize_callback' => array( $this, 'sanitize_yesno' ),
            'default'           => 'no',
        ) );

        register_setting( 'of_tb_settings', self::OPT_DATE_FORMAT, array(
            'type'              => 'string',
            'sanitize_callback' => array( $this, 'sanitize_date_format' ),
            'default'           => 'wp',
        ) );
    }

    public function sanitize_mode( $val ) {
        return in_array( $val, array( 'per_user', 'shared' ), true ) ? $val : 'per_user';
    }

    public function sanitize_yesno( $val ) {
        return ( $val === 'no' ) ? 'no' : 'yes';
    }

    public function sanitize_date_format( $val ) {
        $val = sanitize_key( (string) $val );
        $allowed = array( 'wp', 'dmy', 'mdy', 'ymd' );
        return in_array( $val, $allowed, true ) ? $val : 'wp';
    }


    public function sanitize_widget_limit( $val ) {
        $n = (int) $val;
        if ( $n < 3 ) {
            $n = 3;
        }
        if ( $n > 12 ) {
            $n = 12;
        }
        return $n;
    }

    public function default_categories() {
        return array(
            'general'  => array( 'label' => 'General',  'color' => '#e5e7eb' ),
            'updates'  => array( 'label' => 'Updates',  'color' => '#bfdbfe' ),
            'security' => array( 'label' => 'Security', 'color' => '#fecaca' ),
            'content'  => array( 'label' => 'Content',  'color' => '#bbf7d0' ),
            'seo'      => array( 'label' => 'SEO',      'color' => '#fde68a' ),
            'hosting'  => array( 'label' => 'Hosting',  'color' => '#ddd6fe' ),
        );
    }

    public function sanitize_categories( $val ) {
        if ( ! is_array( $val ) ) {
            return $this->default_categories();
        }

        $clean = array();
        foreach ( $val as $slug => $cat ) {
            $slug = sanitize_key( $slug );
            if ( $slug === '' ) {
                continue;
            }

            $label = isset( $cat['label'] ) ? sanitize_text_field( $cat['label'] ) : ucfirst( $slug );
            $color = isset( $cat['color'] ) ? sanitize_text_field( $cat['color'] ) : '#e5e7eb';

            if ( ! preg_match( '/^#([A-Fa-f0-9]{6})$/', $color ) ) {
                $color = '#e5e7eb';
            }

            $clean[ $slug ] = array(
                'label' => $label,
                'color' => $color,
            );
        }

        if ( empty( $clean ) ) {
            $clean = $this->default_categories();
        }

        return $clean;
    }

    private function allow_assign() {
        return get_option( self::OPT_ASSIGN, 'yes' ) === 'yes';
    }

    private function enable_notices() {
        return get_option( self::OPT_NOTICES, 'yes' ) === 'yes';
    }

    private function get_categories() {
        $cats = get_option( self::OPT_CATEGORIES, $this->default_categories() );
        return is_array( $cats ) ? $cats : $this->default_categories();
    }

    /* -----------------------------
     * Storage
     * ----------------------------- */

    private function storage_base() {
        $mode = get_option( self::OPT_MODE, 'per_user' );
        if ( $mode === 'shared' ) {
            return 'of_tb_shared';
        }
        return 'of_tb_user_' . (int) get_current_user_id();
    }

    private function key_tasks() {
        return $this->storage_base() . '_tasks';
    }

    private function key_log() {
        return $this->storage_base() . '_log';
    }

    private function key_templates() {
        return $this->storage_base() . '_templates';
    }

    private function get_tasks() {
        $tasks = get_option( $this->key_tasks(), array() );
        return is_array( $tasks ) ? $tasks : array();
    }

    private function save_tasks( $tasks ) {
        update_option( $this->key_tasks(), $tasks, false );
    }

    private function get_log() {
        $log = get_option( $this->key_log(), array() );
        return is_array( $log ) ? $log : array();
    }

    private function add_log( $action, $task_title ) {
        $user = wp_get_current_user();
        $log  = $this->get_log();

        $log[] = array(
            'when'   => current_time( 'mysql' ),
            'user'   => $user ? $user->user_login : '',
            'action' => sanitize_text_field( $action ),
            'task'   => sanitize_text_field( $task_title ),
        );

        if ( count( $log ) > 400 ) {
            $log = array_slice( $log, -400 );
        }

        update_option( $this->key_log(), $log, false );
    }

    private function get_user_templates() {
        $tpls = get_option( $this->key_templates(), array() );
        return is_array( $tpls ) ? $tpls : array();
    }

    private function save_user_templates( $tpls ) {
        update_option( $this->key_templates(), $tpls, false );
    }

    private function back_url( $tab = 'board', $extra = array() ) {
        $url = admin_url( 'admin.php?page=otterfixer-task-board&tab=' . rawurlencode( $tab ) );
        if ( ! empty( $extra ) ) {
            $url = add_query_arg( $extra, $url );
        }
        return $url;
    }

    private function today() {
        return date_i18n( 'Y-m-d' );
    }

    private function new_id() {
        return (string) time() . '_' . (string) wp_rand( 1000, 9999 );
    }

    /* -----------------------------
     * Assets
     * ----------------------------- */

    public function enqueue_admin_assets( $hook ) {
        if ( ! is_admin() ) {
            return;
        }

        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        $is_tb_page  = ( $screen && isset( $screen->base ) && $screen->base === 'toplevel_page_otterfixer-task-board' );
        $is_dashboard = ( $screen && isset( $screen->id ) && $screen->id === 'dashboard' );

        if ( ! $is_tb_page && ! $is_dashboard ) {
            return;
        }

        // Sortable for the board page.
        wp_enqueue_script( 'jquery-ui-sortable' );
        wp_enqueue_script( 'jquery-ui-datepicker' );
        wp_enqueue_style( 'wp-jquery-ui-dialog' );

        $nonce = wp_create_nonce( 'of_tb_reorder' );

        $df_key = $this->effective_date_format_key();
        $ui_map = array(
            'dmy' => 'dd/mm/yy',
            'mdy' => 'mm/dd/yy',
            'ymd' => 'yy-mm-dd',
        );
        $ui_format = isset( $ui_map[ $df_key ] ) ? $ui_map[ $df_key ] : 'dd/mm/yy';

        $inline_js = "
        jQuery(function($){
            var list = $('#of-tb-sortable');
            if(list.length){
                list.sortable({
                    handle: '.of-tb-drag',
                    update: function(){
                        var order = [];
                        list.find('li').each(function(){
                            order.push($(this).data('id'));
                        });
                        $.post(ajaxurl, { action:'of_tb_reorder', nonce:'{$nonce}', order:order });
                    }
                });
            }

            $(document).on('click', '.of-tb-notes-toggle', function(e){
                e.preventDefault();
                var id = $(this).data('target');
                $('#' + id).toggle();
            });

            function toggleBulk(){
                var val = $('select[name=\"bulk_action\"]').val();
                if(val){
                    $('.of-tb-wrap').addClass('of-tb-bulk-active');
                } else {
                    $('.of-tb-wrap').removeClass('of-tb-bulk-active');
                    $('#of-tb-select-all').prop('checked', false);
                    $('.of-tb-bulk-item').prop('checked', false);
                }
            }

            // FIX: use delegated handler correctly
            $(document).on('change', 'select[name=\"bulk_action\"]', toggleBulk);

            // If saving as template, require at least one selected task, prompt for a name,
            // then submit immediately so the template is created in one step.
            $(document).on('change', 'select[name=\"bulk_action\"]', function(){
                var val = $(this).val();
                var wrap = $(this).closest('.of-tb-wrap');
                var form = wrap.find('#of-tb-bulk-form');

                if(val === 'save_template'){
                    var anyChecked = wrap.find('input.of-tb-bulk-item:checked').length > 0;
                    if(!anyChecked){
                        window.alert('Please tick at least one task first.');
                        $(this).val('');
                        toggleBulk();
                        return;
                    }

                    var current = form.find('input[name=\"bulk_tpl_name\"]').val();
                    if(!current){
                        var name = window.prompt('Template name');
                        if(name === null){ // cancelled
                            $(this).val('');
                            form.find('input[name=\"bulk_tpl_name\"]').val('');
                            form.find('input[name=\"bulk_tpl_desc\"]').val('');
                            toggleBulk();
                            return;
                        }
                        name = $.trim(name);
                        if(!name){
                            $(this).val('');
                            toggleBulk();
                            return;
                        }
                        form.find('input[name=\"bulk_tpl_name\"]').val(name);
                    }

                    // Auto-submit so user does not have to click Apply.
                    form.trigger('submit');
                } else {
                    form.find('input[name=\"bulk_tpl_name\"]').val('');
                    form.find('input[name=\"bulk_tpl_desc\"]').val('');
                }
            });

            // Collect checked task IDs into the bulk form on submit (avoids nested form issues)
            $(document).on('submit', '#of-tb-bulk-form', function(e){
                var form = $(this);
                var wrap = form.closest('.of-tb-wrap');

                // Remove previously injected IDs
                form.find('input[data-of-tb-dyn=\"1\"]').remove();

                wrap.find('input.of-tb-bulk-item:checked').each(function(){
                    $('<input>').attr({type:'hidden', name:'task_ids[]', 'data-of-tb-dyn':'1'}).val($(this).val()).appendTo(form);
                });

                if(form.find('input[name=\"task_ids[]\"]').length === 0){
                    e.preventDefault();
                    window.alert('Please tick at least one task first.');
                    return false;
                }
            });

            $(document).on('change', '#of-tb-select-all', function(){
                var checked = $(this).is(':checked');
                var wrap = $(this).closest('.of-tb-wrap');
                wrap.find('input.of-tb-bulk-item').prop('checked', checked);
            });



            // Date picker for due dates (respects settings).
            var ofTbDateFormat = '{$ui_format}';
            if ( $.fn.datepicker ) {
                $('input.of-tb-date').attr('autocomplete','off').datepicker({
                    dateFormat: ofTbDateFormat
                });
            }

            toggleBulk();
        });
        ";
        wp_add_inline_script( 'jquery-ui-sortable', $inline_js );

        $inline_css = "
        .of-tb-wrap{margin-top:10px;}
        .of-tb-top{display:flex; gap:12px; flex-wrap:wrap; align-items:flex-start;}
        .of-tb-card{background:#fff; border:1px solid #e5e7eb; border-radius:14px; padding:14px; box-shadow:0 1px 2px rgba(0,0,0,0.04);}
        .of-tb-card h2{margin:0 0 10px; font-size:14px;}
        .of-tb-add-grid{display:flex; gap:10px; flex-wrap:wrap; align-items:center;}
        .of-tb-add-grid input[type=text]{width:360px; max-width:100%;}
        .of-tb-add-grid input[type=date]{width:150px;}
        .of-tb-add-grid select{max-width:220px;}
        .of-tb-pill{display:inline-block; padding:3px 10px; border-radius:999px; border:1px solid #e5e7eb; font-size:12px; line-height:1.4; background:#fff;}
        .of-tb-title{font-size:14px; font-weight:600;}
        .of-tb-done .of-tb-title{text-decoration:line-through; opacity:0.6;}
        .of-tb-meta{margin-top:6px; display:flex; flex-wrap:wrap; gap:6px; align-items:center;}
        .of-tb-notes{display:none; margin-top:8px; padding:10px; background:#fafafa; border:1px solid #eee; border-radius:12px;}
        .of-tb-overdue{color:#b32d2e;}
	        .of-tb-actions{display:flex; gap:14px; align-items:center; flex-wrap:wrap;}
	        .of-tb-actions .button-link-delete{margin-left:2px;}
	        .of-tb-tpl-actions{gap:10px; margin-top:10px;}
	        .of-tb-tpl-actions form{margin:0;}
	        .of-tb-tpl-actions .button-link-delete{margin-left:0;}
        .of-tb-mini{font-size:12px; opacity:0.85;}
        .of-tb-tabs a{margin-right:10px; text-decoration:none;}
        .of-tb-tabs .nav-tab{border-radius:10px 10px 0 0;}

        /* Task list */
        .of-tb-list{margin:10px 0 0;}
        .of-tb-list li{display:flex; gap:10px; align-items:flex-start; padding:12px 0; border-bottom:1px solid #f0f0f0;}

        /* Drag handle: fixed width so every row lines up */
        .of-tb-drag{cursor:move; user-select:none; font-size:16px; line-height:1; opacity:0.7; padding:3px 7px; border:1px solid #eee; border-radius:10px; width:28px; flex:0 0 28px; text-align:center;}

        /* Bulk checkbox hidden unless bulk mode is active */
        .of-tb-bulk-item{display:none; width:18px; flex:0 0 18px; margin-top:2px;}
        .of-tb-bulk-active .of-tb-bulk-item{display:inline-block;}

        /* Done toggle wrapper keeps alignment */
        .of-tb-done-toggle{width:34px; flex:0 0 34px; display:flex; align-items:flex-start; justify-content:center;}

        /* Tick button (action state, not just status) */
        .of-tb-done-btn{width:28px; height:28px; border:1px solid #fecaca; border-radius:10px; background:#fff; cursor:pointer; display:flex; align-items:center; justify-content:center; padding:0;}
        .of-tb-done-btn:hover{background:#fef2f2;}
        .of-tb-done-btn .dashicons{font-size:18px; line-height:1; color:#dc2626;}
        .of-tb-done-btn.is-done{background:#f0fdf4; border-color:#bbf7d0;}
        .of-tb-done-btn.is-done .dashicons{color:#16a34a;}

        /* Template cards */
        .of-tb-tpl-grid{display:flex; gap:12px; flex-wrap:wrap;}
        .of-tb-tpl-card{width:360px; max-width:100%;}
        .of-tb-badge{display:inline-block; padding:2px 8px; border-radius:999px; font-size:11px; border:1px solid #e5e7eb; background:#f8fafc;}
        ";

        wp_register_style( 'of-tb-admin', false, array(), '1.1.0' );
        wp_enqueue_style( 'of-tb-admin' );
        wp_add_inline_style( 'of-tb-admin', $inline_css );
    }

    /* -----------------------------
     * Dashboard widget
     * ----------------------------- */

    public function register_dashboard_widget() {
        wp_add_dashboard_widget(
            'of_tb_quick_tasks',
            __( 'Quick Tasks', 'otterfixer-task-board' ),
            array( $this, 'render_dashboard_widget' )
        );
    }

    public function render_dashboard_widget() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            echo '<p>' . esc_html__( 'You do not have permission to view this.', 'otterfixer-task-board' ) . '</p>';
            return;
        }

        $limit = (int) get_option( self::OPT_WIDGET_LIMIT, 5 );
        $tasks = $this->get_tasks();
        $cats  = $this->get_categories();

        $items = $this->sorted_tasks_for_view( $tasks, false );

        echo '<div class="of-tb-wrap">';
        echo '<p class="of-tb-mini">' . esc_html__( 'A quick view. Open Task Board for the full manager.', 'otterfixer-task-board' ) . '</p>';

        // Quick add
        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="margin:10px 0 6px;">';
        wp_nonce_field( 'of_tb_add', 'of_tb_nonce' );
        echo '<input type="hidden" name="action" value="of_tb_add" />';
        echo '<input type="hidden" name="from_tab" value="dashboard" />';

        echo '<div class="of-tb-add-grid">';
        echo '<input type="text" name="task_title" placeholder="' . esc_attr__( 'Add a task...', 'otterfixer-task-board' ) . '" required />';
        echo '<button class="button button-primary" type="submit">' . esc_html__( 'Add', 'otterfixer-task-board' ) . '</button>';
        echo '</div>';

        echo '</form>';

        if ( empty( $items ) ) {
            echo '<p>' . esc_html__( 'No tasks yet.', 'otterfixer-task-board' ) . '</p>';
        } else {
            echo '<ul class="of-tb-list">';
            $shown = 0;
            foreach ( $items as $id => $t ) {
                if ( $shown >= $limit ) {
                    break;
                }
                $shown++;

                $title = isset( $t['title'] ) ? (string) $t['title'] : '';
                $done  = ! empty( $t['done'] );
                $due   = isset( $t['due'] ) ? (string) $t['due'] : '';
                $cat   = isset( $t['category'] ) ? (string) $t['category'] : 'general';

                $cat_label = isset( $cats[ $cat ]['label'] ) ? (string) $cats[ $cat ]['label'] : ucfirst( $cat );
                $cat_color = isset( $cats[ $cat ]['color'] ) ? (string) $cats[ $cat ]['color'] : '#e5e7eb';

                $overdue = ( ! $done && $due && $due < $this->today() );

                echo '<li class="' . esc_attr( $done ? 'of-tb-done' : '' ) . '">';
                echo '<div style="flex:1;">';
                echo '<div class="of-tb-title">' . esc_html( $title ) . '</div>';
                echo '<div class="of-tb-meta">';
                echo '<span class="of-tb-pill" style="background:' . esc_attr( $cat_color ) . ';">' . esc_html( $cat_label ) . '</span>';
                if ( $due ) {
                    echo '<span class="of-tb-pill ' . esc_attr( $overdue ? 'of-tb-overdue' : '' ) . '">' . esc_html__( 'Due:', 'otterfixer-task-board' ) . ' ' . esc_html( $this->display_date( $due ) );
                    if ( $overdue ) {
                        echo ' ' . esc_html__( '(overdue)', 'otterfixer-task-board' );
                    }
                    echo '</span>';
                }
                echo '</div>';
                echo '</div>';
                echo '</li>';
            }
            echo '</ul>';
        }

        echo '<p style="margin-top:10px;"><a class="button" href="' . esc_url( $this->back_url( 'board' ) ) . '">' . esc_html__( 'Open Task Board', 'otterfixer-task-board' ) . '</a></p>';
        echo '</div>';
    }

    /* -----------------------------
     * Page render
     * ----------------------------- */

    public function render_page() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }

        $tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'board'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( ! in_array( $tab, array( 'board', 'completed', 'templates', 'log', 'settings' ), true ) ) {
            $tab = 'board';
        }

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__( 'Task Board', 'otterfixer-task-board' ) . '</h1>';

        echo '<h2 class="nav-tab-wrapper of-tb-tabs">';
        $this->tab_link( 'board', __( 'Board', 'otterfixer-task-board' ), $tab );
        $this->tab_link( 'completed', __( 'Completed', 'otterfixer-task-board' ), $tab );
        $this->tab_link( 'templates', __( 'Templates', 'otterfixer-task-board' ), $tab );
        $this->tab_link( 'log', __( 'Activity Log', 'otterfixer-task-board' ), $tab );
        if ( current_user_can( self::CAP_MANAGE ) ) {
            $this->tab_link( 'settings', __( 'Settings', 'otterfixer-task-board' ), $tab );
        }
        echo '</h2>';

        switch ( $tab ) {
            case 'completed':
                $this->render_completed_tab();
                break;
            case 'templates':
                $this->render_templates_tab();
                break;
            case 'log':
                $this->render_log_tab();
                break;
            case 'settings':
                $this->render_settings_tab();
                break;
            case 'board':
            default:
                $this->render_board_tab();
                break;
        }

        echo '</div>';
    }

    private function tab_link( $key, $label, $active ) {
        $cls = ( $key === $active ) ? 'nav-tab nav-tab-active' : 'nav-tab';
        echo '<a class="' . esc_attr( $cls ) . '" href="' . esc_url( $this->back_url( $key ) ) . '">' . esc_html( $label ) . '</a>';
    }

    private function render_board_tab() {
        $tasks = $this->get_tasks();
        $cats  = $this->get_categories();

        // Filters
        $q     = isset( $_GET['q'] ) ? sanitize_text_field( wp_unslash( $_GET['q'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $f_cat = isset( $_GET['cat'] ) ? sanitize_key( wp_unslash( $_GET['cat'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $f_pri = isset( $_GET['pri'] ) ? sanitize_key( wp_unslash( $_GET['pri'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $f_ass = isset( $_GET['ass'] ) ? (int) $_GET['ass'] : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

        $items = $this->sorted_tasks_for_view( $tasks, false );

        // Apply filters
        if ( $q !== '' || $f_cat !== '' || $f_pri !== '' || $f_ass > 0 ) {
            $filtered = array();
            foreach ( $items as $id => $t ) {
                if ( ! empty( $t['done'] ) ) {
                    continue;
                }

                $title = isset( $t['title'] ) ? (string) $t['title'] : '';
                $notes = isset( $t['notes'] ) ? (string) $t['notes'] : '';
                $cat   = isset( $t['category'] ) ? (string) $t['category'] : 'general';
                $pri   = isset( $t['priority'] ) ? (string) $t['priority'] : 'normal';
                $ass   = isset( $t['assignee'] ) ? (int) $t['assignee'] : 0;

                if ( $q !== '' ) {
                    $hay = strtolower( $title . ' ' . $notes );
                    if ( strpos( $hay, strtolower( $q ) ) === false ) {
                        continue;
                    }
                }
                if ( $f_cat !== '' && $cat !== $f_cat ) {
                    continue;
                }
                if ( $f_pri !== '' && $pri !== $f_pri ) {
                    continue;
                }
                if ( $f_ass > 0 && $ass !== $f_ass ) {
                    continue;
                }

                $filtered[ $id ] = $t;
            }
            $items = $filtered;
        }

        // Stats
        $stats = $this->stats_summary( $tasks );

        echo '<div class="of-tb-wrap">';
        echo '<div class="of-tb-top">';

        // Add card
        echo '<div class="of-tb-card" style="flex:2; min-width:520px;">';
        echo '<h2>' . esc_html__( 'Add Task', 'otterfixer-task-board' ) . '</h2>';
        $this->render_add_form( 'board' );
        echo '</div>';

        // Stats card
        echo '<div class="of-tb-card" style="flex:1; min-width:280px;">';
        echo '<h2>' . esc_html__( 'Overview', 'otterfixer-task-board' ) . '</h2>';
        echo '<p><span class="of-tb-pill">' . esc_html__( 'Open:', 'otterfixer-task-board' ) . ' ' . esc_html( (string) $stats['open'] ) . '</span> ';
        echo '<span class="of-tb-pill">' . esc_html__( 'Overdue:', 'otterfixer-task-board' ) . ' ' . esc_html( (string) $stats['overdue'] ) . '</span> ';
        echo '<span class="of-tb-pill">' . esc_html__( 'Completed:', 'otterfixer-task-board' ) . ' ' . esc_html( (string) $stats['done'] ) . '</span></p>';

        echo '<p class="of-tb-mini">' . esc_html__( 'Tip: Save your own templates, then apply them any time.', 'otterfixer-task-board' ) . '</p>';
        echo '<p class="of-tb-actions">';
        echo '<a class="button" href="' . esc_url( $this->back_url( 'templates' ) ) . '">' . esc_html__( 'Templates', 'otterfixer-task-board' ) . '</a>';
        echo '<a class="button" href="' . esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=of_tb_export_tasks' ), 'of_tb_export_tasks', 'of_tb_nonce' ) ) . '">' . esc_html__( 'Export tasks CSV', 'otterfixer-task-board' ) . '</a>';
        echo '</p>';
        echo '</div>';

        echo '</div>';

        // Filters + list card
        echo '<div class="of-tb-card" style="margin-top:12px;">';
        echo '<h2>' . esc_html__( 'Tasks', 'otterfixer-task-board' ) . '</h2>';

        $this->render_filters_bar( $cats, $q, $f_cat, $f_pri, $f_ass );

        if ( empty( $items ) ) {
            echo '<p>' . esc_html__( 'No matching tasks. Try clearing filters or add a new one above.', 'otterfixer-task-board' ) . '</p>';
        } else {
            $this->render_tasks_list( $items, $cats, false );
        }

        echo '</div>';
        echo '</div>';
    }

    private function render_completed_tab() {
        $tasks = $this->get_tasks();
        $cats  = $this->get_categories();

        $done_items = $this->sorted_tasks_for_view( $tasks, true );

        echo '<div class="of-tb-wrap">';
        echo '<div class="of-tb-card">';
        echo '<h2>' . esc_html__( 'Completed Tasks', 'otterfixer-task-board' ) . '</h2>';

        if ( empty( $done_items ) ) {
            echo '<p>' . esc_html__( 'Nothing completed yet.', 'otterfixer-task-board' ) . '</p>';
        } else {
            $this->render_tasks_list( $done_items, $cats, true );
        }

        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="margin-top:12px;">';
        wp_nonce_field( 'of_tb_clear_completed', 'of_tb_nonce' );
        echo '<input type="hidden" name="action" value="of_tb_clear_completed" />';
        echo '<button class="button" type="submit">' . esc_html__( 'Clear completed', 'otterfixer-task-board' ) . '</button>';
        echo '</form>';

        echo '</div></div>';
    }

    private function render_templates_tab() {
        $templates = $this->templates();
        $user_tpls = $this->get_user_templates();

        $draft = get_transient( self::TRANSIENT_TEMPLATE_DRAFT . (int) get_current_user_id() );

        echo '<div class="of-tb-wrap">';
        echo '<div class="of-tb-card">';
        echo '<h2>' . esc_html__( 'Templates', 'otterfixer-task-board' ) . '</h2>';
        echo '<p class="of-tb-mini">' . esc_html__( 'Use built in templates, or create your own and apply them whenever you need.', 'otterfixer-task-board' ) . '</p>';

        // Create template from selected tasks (draft)
        echo '<div class="of-tb-card" style="margin-top:12px; max-width:980px;">';
        echo '<h2>' . esc_html__( 'Create a template', 'otterfixer-task-board' ) . '</h2>';

        if ( is_array( $draft ) && ! empty( $draft['items'] ) ) {
            $count = count( $draft['items'] );
            /* translators: %d: Number of tasks selected to save as a template. */
            echo '<p class="of-tb-mini">' . esc_html( sprintf( __( 'You have %d task(s) ready to save as a template.', 'otterfixer-task-board' ), (int) $count ) ) . '</p>';

            echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
            wp_nonce_field( 'of_tb_create_user_template', 'of_tb_nonce' );
            echo '<input type="hidden" name="action" value="of_tb_create_user_template" />';

            echo '<div class="of-tb-add-grid">';
            echo '<input type="text" name="tpl_name" placeholder="' . esc_attr__( 'Template name', 'otterfixer-task-board' ) . '" required />';
            echo '<input type="text" name="tpl_desc" placeholder="' . esc_attr__( 'Description (optional)', 'otterfixer-task-board' ) . '" style="width:320px; max-width:100%;" />';
            echo '<button class="button button-primary" type="submit">' . esc_html__( 'Save template', 'otterfixer-task-board' ) . '</button>';
            echo '</div>';

            echo '</form>';
        } else {
            echo '<p class="of-tb-mini">' . esc_html__( 'Tip: On the Board tab, choose a bulk action called "Save as template" to build a template from selected tasks.', 'otterfixer-task-board' ) . '</p>';
        }
        echo '</div>';

        // Built in templates
        echo '<h2 style="margin-top:16px;">' . esc_html__( 'Built in', 'otterfixer-task-board' ) . ' <span class="of-tb-badge">' . esc_html__( 'Read only', 'otterfixer-task-board' ) . '</span></h2>';
        echo '<div class="of-tb-tpl-grid">';
        foreach ( $templates as $key => $tpl ) {
            echo '<div class="of-tb-card of-tb-tpl-card">';
            echo '<h2>' . esc_html( $tpl['name'] ) . '</h2>';
            echo '<p class="of-tb-mini">' . esc_html( $tpl['desc'] ) . '</p>';
            echo '<p class="of-tb-mini"><strong>' . esc_html__( 'Tasks:', 'otterfixer-task-board' ) . '</strong> ' . esc_html( (string) count( $tpl['tasks'] ) ) . '</p>';

            echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
            wp_nonce_field( 'of_tb_add_template', 'of_tb_nonce' );
            echo '<input type="hidden" name="action" value="of_tb_add_template" />';
            echo '<input type="hidden" name="template_key" value="' . esc_attr( $key ) . '" />';
            echo '<button class="button button-primary" type="submit">' . esc_html__( 'Add to board', 'otterfixer-task-board' ) . '</button>';
            echo '</form>';
            echo '</div>';
        }
        echo '</div>';

        // User templates
        echo '<h2 style="margin-top:18px;">' . esc_html__( 'My templates', 'otterfixer-task-board' ) . '</h2>';

        if ( empty( $user_tpls ) ) {
            echo '<p class="of-tb-mini">' . esc_html__( 'No templates yet. Use the bulk action on the Board to save selected tasks as a template.', 'otterfixer-task-board' ) . '</p>';
        } else {
            echo '<div class="of-tb-tpl-grid">';
            foreach ( $user_tpls as $tpl_id => $tpl ) {
                $name = isset( $tpl['name'] ) ? (string) $tpl['name'] : '';
                $desc = isset( $tpl['desc'] ) ? (string) $tpl['desc'] : '';
                $items = isset( $tpl['items'] ) && is_array( $tpl['items'] ) ? $tpl['items'] : array();

                echo '<div class="of-tb-card of-tb-tpl-card">';
                echo '<h2>' . esc_html( $name ) . '</h2>';
                if ( $desc !== '' ) {
                    echo '<p class="of-tb-mini">' . esc_html( $desc ) . '</p>';
                }
                echo '<p class="of-tb-mini"><strong>' . esc_html__( 'Tasks:', 'otterfixer-task-board' ) . '</strong> ' . esc_html( (string) count( $items ) ) . '</p>';

                echo '<div class="of-tb-actions of-tb-tpl-actions">';
                // apply
                echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="display:inline-block; margin:0;">';
                wp_nonce_field( 'of_tb_apply_user_template', 'of_tb_nonce' );
                echo '<input type="hidden" name="action" value="of_tb_apply_user_template" />';
                echo '<input type="hidden" name="tpl_id" value="' . esc_attr( $tpl_id ) . '" />';
                echo '<button class="button button-primary" type="submit">' . esc_html__( 'Add to board', 'otterfixer-task-board' ) . '</button>';
                echo '</form>';

                // delete
                echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="display:inline-block; margin:0;">';
                wp_nonce_field( 'of_tb_delete_user_template', 'of_tb_nonce' );
                echo '<input type="hidden" name="action" value="of_tb_delete_user_template" />';
                echo '<input type="hidden" name="tpl_id" value="' . esc_attr( $tpl_id ) . '" />';
                echo '<button class="button button-link-delete" type="submit">' . esc_html__( 'Delete', 'otterfixer-task-board' ) . '</button>';
                echo '</form>';

                echo '</div>';
                echo '</div>';
            }
            echo '</div>';
        }

        // Export/Import
        echo '<div class="of-tb-card" style="margin-top:14px; max-width:980px;">';
        echo '<h2>' . esc_html__( 'Export and import', 'otterfixer-task-board' ) . '</h2>';
        echo '<div class="of-tb-actions">';

        echo '<a class="button" href="' . esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=of_tb_export_templates' ), 'of_tb_export_templates', 'of_tb_nonce' ) ) . '">' . esc_html__( 'Export templates CSV', 'otterfixer-task-board' ) . '</a>';

        echo '<a class="button" href="' . esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=of_tb_download_templates_template' ), 'of_tb_download_templates_template', 'of_tb_nonce' ) ) . '">' . esc_html__( 'Download CSV template', 'otterfixer-task-board' ) . '</a>';

        echo '</div>';

        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" enctype="multipart/form-data">';
        wp_nonce_field( 'of_tb_import_templates', 'of_tb_nonce' );
        echo '<input type="hidden" name="action" value="of_tb_import_templates" />';
        echo '<input type="file" name="tpl_csv" accept=".csv" />';
        echo '<button class="button" type="submit">' . esc_html__( 'Import CSV', 'otterfixer-task-board' ) . '</button>';
        echo '<p class="of-tb-mini">' . esc_html__( 'Import will add templates to your list. Existing templates are not overwritten.', 'otterfixer-task-board' ) . '</p>';
        echo '</form>';

        echo '</div>';

        echo '</div></div>';
    }

    private function render_log_tab() {
        $log = array_reverse( $this->get_log() );

        echo '<div class="of-tb-wrap">';
        echo '<div class="of-tb-card">';
        echo '<h2>' . esc_html__( 'Activity Log', 'otterfixer-task-board' ) . '</h2>';
        echo '<p class="of-tb-actions">';
        echo '<a class="button" href="' . esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=of_tb_export_log' ), 'of_tb_export_log', 'of_tb_nonce' ) ) . '">' . esc_html__( 'Export log CSV', 'otterfixer-task-board' ) . '</a>';
        echo '</p>';

        if ( empty( $log ) ) {
            echo '<p>' . esc_html__( 'No activity yet.', 'otterfixer-task-board' ) . '</p>';
        } else {
            echo '<table class="widefat striped" style="margin-top:10px;">';
            echo '<thead><tr>';
            echo '<th>' . esc_html__( 'When', 'otterfixer-task-board' ) . '</th>';
            echo '<th>' . esc_html__( 'User', 'otterfixer-task-board' ) . '</th>';
            echo '<th>' . esc_html__( 'Action', 'otterfixer-task-board' ) . '</th>';
            echo '<th>' . esc_html__( 'Task', 'otterfixer-task-board' ) . '</th>';
            echo '</tr></thead><tbody>';

            $count = 0;
            foreach ( $log as $row ) {
                $count++;
                if ( $count > 200 ) {
                    break;
                }

                echo '<tr>';
                $when_raw = isset( $row['when'] ) ? (string) $row['when'] : '';
                echo '<td>' . esc_html( $this->display_datetime( $when_raw ) ) . '</td>';
                echo '<td>' . esc_html( isset( $row['user'] ) ? (string) $row['user'] : '' ) . '</td>';
                echo '<td>' . esc_html( isset( $row['action'] ) ? (string) $row['action'] : '' ) . '</td>';
                echo '<td>' . esc_html( isset( $row['task'] ) ? (string) $row['task'] : '' ) . '</td>';
                echo '</tr>';
            }

            echo '</tbody></table>';
            echo '<p class="of-tb-mini" style="margin-top:10px;">' . esc_html__( 'Showing the most recent 200 entries.', 'otterfixer-task-board' ) . '</p>';
        }

        echo '</div></div>';
    }

    private function render_settings_tab() {
        if ( ! current_user_can( self::CAP_MANAGE ) ) {
            echo '<div class="of-tb-wrap"><div class="of-tb-card"><p>' . esc_html__( 'You do not have permission to view settings.', 'otterfixer-task-board' ) . '</p></div></div>';
            return;
        }

        $mode   = get_option( self::OPT_MODE, 'per_user' );
        $assign = get_option( self::OPT_ASSIGN, 'yes' );
        $notice = get_option( self::OPT_NOTICES, 'yes' );
        $delopt = get_option( self::OPT_DELETE_DATA, 'no' );
        $df     = get_option( self::OPT_DATE_FORMAT, 'wp' );
        $limit  = (int) get_option( self::OPT_WIDGET_LIMIT, 5 );
        $cats   = $this->get_categories();

        echo '<div class="of-tb-wrap"><div class="of-tb-card" style="max-width:860px;">';
        echo '<h2>' . esc_html__( 'Settings', 'otterfixer-task-board' ) . '</h2>';

        echo '<form method="post" action="options.php">';
        settings_fields( 'of_tb_settings' );

        echo '<table class="form-table" role="presentation">';

        echo '<tr><th scope="row">' . esc_html__( 'List mode', 'otterfixer-task-board' ) . '</th><td>';
        echo '<label style="display:block; margin-bottom:6px;"><input type="radio" name="' . esc_attr( self::OPT_MODE ) . '" value="per_user" ' . checked( $mode, 'per_user', false ) . ' /> ' . esc_html__( 'Per user', 'otterfixer-task-board' ) . '</label>';
        echo '<label style="display:block;"><input type="radio" name="' . esc_attr( self::OPT_MODE ) . '" value="shared" ' . checked( $mode, 'shared', false ) . ' /> ' . esc_html__( 'Shared for all users', 'otterfixer-task-board' ) . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Date format', 'otterfixer-task-board' ) . '</th><td>';
        echo '<select name="' . esc_attr( self::OPT_DATE_FORMAT ) . '">';
        echo '<option value="wp"' . selected( $df, 'wp', false ) . '>' . esc_html__( 'Use WordPress setting', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="dmy"' . selected( $df, 'dmy', false ) . '>' . esc_html__( 'DD/MM/YYYY', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="mdy"' . selected( $df, 'mdy', false ) . '>' . esc_html__( 'MM/DD/YYYY', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="ymd"' . selected( $df, 'ymd', false ) . '>' . esc_html__( 'YYYY-MM-DD', 'otterfixer-task-board' ) . '</option>';
        echo '</select>';
        echo '<p class="description">' . esc_html__( 'Controls how due dates are displayed in the calendar field. Dates are stored safely as YYYY-MM-DD behind the scenes.', 'otterfixer-task-board' ) . '</p>';
        echo '</td></tr>';


                echo '<tr><th scope="row">' . esc_html__( 'Assignments', 'otterfixer-task-board' ) . '</th><td>';
        echo '<label><input type="checkbox" name="' . esc_attr( self::OPT_ASSIGN ) . '" value="yes" ' . checked( $assign, 'yes', false ) . ' /> ' . esc_html__( 'Allow assigning tasks to users', 'otterfixer-task-board' ) . '</label>';
        echo '</td></tr>';

echo '<tr><th scope="row">' . esc_html__( 'Overdue notice', 'otterfixer-task-board' ) . '</th><td>';
        echo '<label><input type="checkbox" name="' . esc_attr( self::OPT_NOTICES ) . '" value="yes" ' . checked( $notice, 'yes', false ) . ' /> ' . esc_html__( 'Show notice on Dashboard when tasks are overdue', 'otterfixer-task-board' ) . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Uninstall behaviour', 'otterfixer-task-board' ) . '</th><td>';
        echo '<label><input type="checkbox" name="' . esc_attr( self::OPT_DELETE_DATA ) . '" value="yes" ' . checked( $delopt, 'yes', false ) . ' /> ' . esc_html__( 'Delete all Task Board data when the plugin is deleted', 'otterfixer-task-board' ) . '</label>';
        echo '<p class="description">' . esc_html__( 'By default WordPress keeps plugin data on deletion. Enable this if you want all tasks, templates and logs removed when you delete the plugin.', 'otterfixer-task-board' ) . '</p>';
        echo '</td></tr>';
echo '<tr><th scope="row">' . esc_html__( 'Dashboard widget limit', 'otterfixer-task-board' ) . '</th><td>';
        echo '<input type="number" min="3" max="12" name="' . esc_attr( self::OPT_WIDGET_LIMIT ) . '" value="' . esc_attr( (string) $limit ) . '" />';
        echo '<p class="description">' . esc_html__( 'How many tasks to show in the Quick Tasks widget.', 'otterfixer-task-board' ) . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__( 'Categories', 'otterfixer-task-board' ) . '</th><td>';
        echo '<p class="description">' . esc_html__( 'Edit labels and colours. Slugs are fixed.', 'otterfixer-task-board' ) . '</p>';
        foreach ( $cats as $slug => $cat ) {
            $label = isset( $cat['label'] ) ? (string) $cat['label'] : ucfirst( $slug );
            $color = isset( $cat['color'] ) ? (string) $cat['color'] : '#e5e7eb';

            echo '<div style="display:flex; gap:10px; align-items:center; margin:8px 0;">';
            echo '<code style="min-width:110px;">' . esc_html( $slug ) . '</code>';
            echo '<input type="text" name="' . esc_attr( self::OPT_CATEGORIES ) . '[' . esc_attr( $slug ) . '][label]" value="' . esc_attr( $label ) . '" style="width:240px;" />';
            echo '<input type="text" name="' . esc_attr( self::OPT_CATEGORIES ) . '[' . esc_attr( $slug ) . '][color]" value="' . esc_attr( $color ) . '" style="width:120px;" />';
            echo '<span style="display:inline-block; width:18px; height:18px; border-radius:999px; background:' . esc_attr( $color ) . '; border:1px solid #ddd;"></span>';
            echo '</div>';
        }
        echo '</td></tr>';

        echo '</table>';

        submit_button( __( 'Save settings', 'otterfixer-task-board' ) );

        echo '</form>';
        echo '</div></div>';
    }

    /* -----------------------------
     * UI helpers
     * ----------------------------- */

    private function render_add_form( $from_tab ) {
        $cats = $this->get_categories();
        $assign_enabled = $this->allow_assign();

        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
        wp_nonce_field( 'of_tb_add', 'of_tb_nonce' );
        echo '<input type="hidden" name="action" value="of_tb_add" />';
        echo '<input type="hidden" name="from_tab" value="' . esc_attr( $from_tab ) . '" />';

        echo '<div class="of-tb-add-grid">';
        echo '<input type="text" name="task_title" placeholder="' . esc_attr__( 'Add a task...', 'otterfixer-task-board' ) . '" required />';
        echo '<input type="text" class="of-tb-date" name="task_due" value="" placeholder="' . esc_attr( $this->date_placeholder() ) . '" />';

        echo '<select name="task_priority">';
        echo '<option value="normal">' . esc_html__( 'Normal', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="high">' . esc_html__( 'High', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="low">' . esc_html__( 'Low', 'otterfixer-task-board' ) . '</option>';
        echo '</select>';

        echo '<select name="task_category">';
        foreach ( $cats as $slug => $cat ) {
            echo '<option value="' . esc_attr( $slug ) . '">' . esc_html( $cat['label'] ) . '</option>';
        }
        echo '</select>';

        echo '<select name="task_recur">';
        echo '<option value="none">' . esc_html__( 'No repeat', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="daily">' . esc_html__( 'Daily', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="weekly">' . esc_html__( 'Weekly', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="monthly">' . esc_html__( 'Monthly', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="quarterly">' . esc_html__( 'Quarterly', 'otterfixer-task-board' ) . '</option>';
        echo '</select>';

        if ( $assign_enabled && current_user_can( 'list_users' ) ) {
            echo '<select name="task_assignee">';
            echo '<option value="0">' . esc_html__( 'Unassigned', 'otterfixer-task-board' ) . '</option>';
            $users = get_users( array( 'fields' => array( 'ID', 'display_name' ) ) );
            foreach ( $users as $u ) {
                echo '<option value="' . esc_attr( (string) $u->ID ) . '">' . esc_html( $u->display_name ) . '</option>';
            }
            echo '</select>';
        }

        echo '<input type="text" name="task_notes" placeholder="' . esc_attr__( 'Notes (optional)', 'otterfixer-task-board' ) . '" style="width:260px; max-width:100%;" />';
        echo '<button class="button button-primary" type="submit">' . esc_html__( 'Add', 'otterfixer-task-board' ) . '</button>';
        echo '</div>';

        echo '</form>';
    }

    private function render_filters_bar( $cats, $q, $f_cat, $f_pri, $f_ass ) {
        $base = admin_url( 'admin.php?page=otterfixer-task-board&tab=board' );

        echo '<form method="get" action="' . esc_url( admin_url( 'admin.php' ) ) . '" style="margin:10px 0;">';
        echo '<input type="hidden" name="page" value="otterfixer-task-board" />';
        echo '<input type="hidden" name="tab" value="board" />';

        echo '<div class="of-tb-actions">';
        echo '<input type="text" name="q" value="' . esc_attr( $q ) . '" placeholder="' . esc_attr__( 'Search tasks...', 'otterfixer-task-board' ) . '" />';

        echo '<select name="cat">';
        echo '<option value="">' . esc_html__( 'All categories', 'otterfixer-task-board' ) . '</option>';
        foreach ( $cats as $slug => $cat ) {
            echo '<option value="' . esc_attr( $slug ) . '" ' . selected( $f_cat, $slug, false ) . '>' . esc_html( $cat['label'] ) . '</option>';
        }
        echo '</select>';

        echo '<select name="pri">';
        echo '<option value="">' . esc_html__( 'All priorities', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="high" ' . selected( $f_pri, 'high', false ) . '>' . esc_html__( 'High', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="normal" ' . selected( $f_pri, 'normal', false ) . '>' . esc_html__( 'Normal', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="low" ' . selected( $f_pri, 'low', false ) . '>' . esc_html__( 'Low', 'otterfixer-task-board' ) . '</option>';
        echo '</select>';

        if ( $this->allow_assign() && current_user_can( 'list_users' ) ) {
            echo '<select name="ass">';
            echo '<option value="0">' . esc_html__( 'All assignees', 'otterfixer-task-board' ) . '</option>';
            $users = get_users( array( 'fields' => array( 'ID', 'display_name' ) ) );
            foreach ( $users as $u ) {
                echo '<option value="' . esc_attr( (string) $u->ID ) . '" ' . selected( $f_ass, (int) $u->ID, false ) . '>' . esc_html( $u->display_name ) . '</option>';
            }
            echo '</select>';
        }

        echo '<button class="button" type="submit">' . esc_html__( 'Filter', 'otterfixer-task-board' ) . '</button>';
        echo '<a class="button" href="' . esc_url( $base ) . '">' . esc_html__( 'Clear', 'otterfixer-task-board' ) . '</a>';
        echo '</div>';
        echo '</form>';
    }

    private function render_tasks_list( $items, $cats, $completed_view = false ) {
        echo '<div class="of-tb-wrap">';

        // Bulk actions wrapper
        echo '<form id="of-tb-bulk-form" method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
        wp_nonce_field( 'of_tb_bulk', 'of_tb_nonce' );
        echo '<input type="hidden" name="action" value="of_tb_bulk" />';
        echo '<input type="hidden" name="return_tab" value="' . esc_attr( $completed_view ? 'completed' : 'board' ) . '" />';

        
        echo '<input type="hidden" name="bulk_tpl_name" id="of-tb-bulk-tpl-name" value="" />';
        echo '<input type="hidden" name="bulk_tpl_desc" id="of-tb-bulk-tpl-desc" value="" />';
echo '<div class="of-tb-actions" style="margin-bottom:8px;">';
        echo '<label class="of-tb-mini"><input type="checkbox" id="of-tb-select-all" /> ' . esc_html__( 'Select all', 'otterfixer-task-board' ) . '</label>';

        echo '<select name="bulk_action">';
        echo '<option value="">' . esc_html__( 'Bulk action...', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="done">' . esc_html__( 'Mark as done', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="undone">' . esc_html__( 'Mark as not done', 'otterfixer-task-board' ) . '</option>';
        echo '<option value="delete">' . esc_html__( 'Delete', 'otterfixer-task-board' ) . '</option>';
        if ( ! $completed_view ) {
            echo '<option value="save_template">' . esc_html__( 'Save as template', 'otterfixer-task-board' ) . '</option>';
        }
        echo '</select>';




        echo '<button class="button" type="submit">' . esc_html__( 'Apply', 'otterfixer-task-board' ) . '</button>';
        echo '<span class="of-tb-mini">' . esc_html__( 'Tip: drag tasks using the handle on the left.', 'otterfixer-task-board' ) . '</span>';
        echo '</div>';

        echo '</form>';

        echo '<ul class="of-tb-list" id="of-tb-sortable">';

        $today = $this->today();

        foreach ( $items as $id => $t ) {
            $title = isset( $t['title'] ) ? (string) $t['title'] : '';
            $done  = ! empty( $t['done'] );
            $due   = isset( $t['due'] ) ? (string) $t['due'] : '';
            $pri   = isset( $t['priority'] ) ? (string) $t['priority'] : 'normal';
            $notes = isset( $t['notes'] ) ? (string) $t['notes'] : '';
            $cat   = isset( $t['category'] ) ? (string) $t['category'] : 'general';
            $recur = isset( $t['recur'] ) ? (string) $t['recur'] : 'none';
            $ass   = isset( $t['assignee'] ) ? (int) $t['assignee'] : 0;

            $cat_label = isset( $cats[ $cat ]['label'] ) ? (string) $cats[ $cat ]['label'] : ucfirst( $cat );
            $cat_color = isset( $cats[ $cat ]['color'] ) ? (string) $cats[ $cat ]['color'] : '#e5e7eb';

            $overdue = ( ! $done && $due && $due < $today );

            $assignee_name = '';
            if ( $ass > 0 ) {
                $u = get_user_by( 'id', $ass );
                if ( $u ) {
                    $assignee_name = $u->display_name;
                }
            }

            $notes_id = 'of_tb_notes_' . preg_replace( '/[^a-zA-Z0-9_]/', '_', (string) $id );

            echo '<li data-id="' . esc_attr( $id ) . '" class="' . esc_attr( $done ? 'of-tb-done' : '' ) . '">';

            echo '<span class="of-tb-drag" title="' . esc_attr__( 'Drag to reorder', 'otterfixer-task-board' ) . '">↕</span>';

            // Bulk checkbox (hidden until a bulk action chosen)
            echo '<input class="of-tb-bulk-item" type="checkbox" name="task_ids[]" value="' . esc_attr( $id ) . '" />';

            // Done toggle (tick button)
            echo '<div class="of-tb-done-toggle">';
            echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="margin:0;">';
            wp_nonce_field( 'of_tb_toggle', 'of_tb_nonce' );
            echo '<input type="hidden" name="action" value="of_tb_toggle" />';
            echo '<input type="hidden" name="task_id" value="' . esc_attr( $id ) . '" />';
            echo '<input type="hidden" name="return_tab" value="' . esc_attr( $completed_view ? 'completed' : 'board' ) . '" />';

            $btn_class = $done ? 'of-tb-done-btn is-done' : 'of-tb-done-btn';
            $btn_label = $done ? __( 'Mark as not done', 'otterfixer-task-board' ) : __( 'Mark as done', 'otterfixer-task-board' );

            echo '<button type="submit" class="' . esc_attr( $btn_class ) . '" aria-label="' . esc_attr( $btn_label ) . '">';
            echo '<span class="dashicons dashicons-yes"></span>';
            echo '</button>';

            echo '</form>';
            echo '</div>';

            echo '<div style="flex:1;">';
            echo '<div class="of-tb-title">' . esc_html( $title ) . '</div>';

            echo '<div class="of-tb-meta">';
            echo '<span class="of-tb-pill" style="background:' . esc_attr( $cat_color ) . ';">' . esc_html( $cat_label ) . '</span>';
            echo '<span class="of-tb-pill">' . esc_html__( 'Priority:', 'otterfixer-task-board' ) . ' ' . esc_html( ucfirst( $pri ) ) . '</span>';

            if ( $due ) {
                echo '<span class="of-tb-pill ' . esc_attr( $overdue ? 'of-tb-overdue' : '' ) . '">' . esc_html__( 'Due:', 'otterfixer-task-board' ) . ' ' . esc_html( $this->display_date( $due ) );
                if ( $overdue ) {
                    echo ' ' . esc_html__( '(overdue)', 'otterfixer-task-board' );
                }
                echo '</span>';
            }

            if ( $recur !== 'none' ) {
                echo '<span class="of-tb-pill">' . esc_html__( 'Repeats:', 'otterfixer-task-board' ) . ' ' . esc_html( ucfirst( $recur ) ) . '</span>';
            }

            if ( $assignee_name ) {
                echo '<span class="of-tb-pill">' . esc_html__( 'Assigned:', 'otterfixer-task-board' ) . ' ' . esc_html( $assignee_name ) . '</span>';
            }

            if ( $notes ) {
                echo '<a href="#" class="of-tb-notes-toggle of-tb-mini" data-target="' . esc_attr( $notes_id ) . '">' . esc_html__( 'View notes', 'otterfixer-task-board' ) . '</a>';
            }

            echo '</div>';

            if ( $notes ) {
                echo '<div class="of-tb-notes" id="' . esc_attr( $notes_id ) . '">' . esc_html( $notes ) . '</div>';
            }

            echo '</div>';

            // Delete
            echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="margin:0;">';
            wp_nonce_field( 'of_tb_delete', 'of_tb_nonce' );
            echo '<input type="hidden" name="action" value="of_tb_delete" />';
            echo '<input type="hidden" name="task_id" value="' . esc_attr( $id ) . '" />';
            echo '<input type="hidden" name="return_tab" value="' . esc_attr( $completed_view ? 'completed' : 'board' ) . '" />';
            echo '<button class="button button-link-delete" type="submit">' . esc_html__( 'Delete', 'otterfixer-task-board' ) . '</button>';
            echo '</form>';

            echo '</li>';
        }

        echo '</ul>';
        echo '</form>';

        echo '</div>';
    }

    private function sorted_tasks_for_view( $tasks, $completed_only ) {
        foreach ( $tasks as $id => $t ) {
            if ( ! isset( $tasks[ $id ]['sort'] ) ) {
                $tasks[ $id ]['sort'] = 0;
            }
        }

        uasort( $tasks, function( $a, $b ) {
            $as = isset( $a['sort'] ) ? (int) $a['sort'] : 0;
            $bs = isset( $b['sort'] ) ? (int) $b['sort'] : 0;
            return $as <=> $bs;
        } );

        $out = array();
        foreach ( $tasks as $id => $t ) {
            $done = ! empty( $t['done'] );
            if ( $completed_only && ! $done ) {
                continue;
            }
            if ( ! $completed_only && $done ) {
                continue;
            }
            $out[ $id ] = $t;
        }
        return $out;
    }

    private function stats_summary( $tasks ) {
        $today   = $this->today();
        $open    = 0;
        $done    = 0;
        $overdue = 0;

        foreach ( $tasks as $t ) {
            $is_done = ! empty( $t['done'] );
            $due = isset( $t['due'] ) ? (string) $t['due'] : '';
            if ( $is_done ) {
                $done++;
            } else {
                $open++;
                if ( $due && $due < $today ) {
                    $overdue++;
                }
            }
        }

        return array(
            'open'    => $open,
            'done'    => $done,
            'overdue' => $overdue,
        );
    }

    /* -----------------------------
     * Notices
     * ----------------------------- */

    public function overdue_notice() {
        if ( ! $this->enable_notices() ) {
            return;
        }
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            return;
        }

        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        if ( ! $screen || $screen->id !== 'dashboard' ) {
            return;
        }

        $tasks = $this->get_tasks();
        if ( empty( $tasks ) ) {
            return;
        }

        $today   = $this->today();
        $overdue = 0;
        foreach ( $tasks as $t ) {
            if ( ! empty( $t['done'] ) ) {
                continue;
            }
            $due = isset( $t['due'] ) ? (string) $t['due'] : '';
            if ( $due && $due < $today ) {
                $overdue++;
            }
        }

        if ( $overdue > 0 ) {
            echo '<div class="notice notice-warning is-dismissible"><p><strong>' . esc_html__( 'Task Board:', 'otterfixer-task-board' ) . '</strong> ';
            /* translators: %d: Number of overdue tasks. */
            echo esc_html( sprintf( _n( '%d task is overdue.', '%d tasks are overdue.', $overdue, 'otterfixer-task-board' ), (int) $overdue ) );
            echo '</p></div>';
        }
    }

    /* -----------------------------
     * Validation and recurrence
     * ----------------------------- */

    private function verify_post_nonce( $action ) {
        $nonce = isset( $_POST['of_tb_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['of_tb_nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, $action ) ) {
            wp_die( esc_html__( 'Security check failed.', 'otterfixer-task-board' ) );
        }
    }

    private function verify_get_nonce( $action ) {
        $nonce = isset( $_GET['of_tb_nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['of_tb_nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, $action ) ) {
            wp_die( esc_html__( 'Security check failed.', 'otterfixer-task-board' ) );
        }
    }

    private function effective_date_format_key() {
        $df = get_option( self::OPT_DATE_FORMAT, 'wp' );
        $df = is_string( $df ) ? $df : 'wp';
        if ( $df !== 'wp' ) {
            return in_array( $df, array( 'dmy', 'mdy', 'ymd' ), true ) ? $df : 'dmy';
        }

        $wp = (string) get_option( 'date_format', 'd/m/Y' );
        $wp = strtolower( $wp );

        // Basic detection for common numeric formats.
        if ( preg_match( '/^[dj].*[\/\-]/', $wp ) ) {
            return 'dmy';
        }
        if ( preg_match( '/^[mn].*[\/\-]/', $wp ) ) {
            return 'mdy';
        }
        if ( preg_match( '/^y.*[\/\-]/', $wp ) ) {
            return 'ymd';
        }

        return 'dmy';
    }

    private function date_placeholder() {
        $key = $this->effective_date_format_key();
        if ( $key === 'mdy' ) {
            return 'mm/dd/yyyy';
        }
        if ( $key === 'ymd' ) {
            return 'yyyy-mm-dd';
        }
        return 'dd/mm/yyyy';
    }

    private function php_date_format() {
        $df = get_option( self::OPT_DATE_FORMAT, 'wp' );
        $df = is_string( $df ) ? $df : 'wp';

        if ( $df === 'wp' ) {
            $fmt = (string) get_option( 'date_format', 'd/m/Y' );
            return $fmt ? $fmt : 'd/m/Y';
        }

        if ( $df === 'mdy' ) {
            return 'm/d/Y';
        }
        if ( $df === 'ymd' ) {
            return 'Y-m-d';
        }
        // default dmy
        return 'd/m/Y';
    }

    private function display_date( $ymd ) {
        $ymd = is_string( $ymd ) ? $ymd : '';
        if ( $ymd === '' ) {
            return '';
        }
        if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $ymd ) ) {
            // Fall back: show raw if we don't recognise the stored format.
            return $ymd;
        }
        $ts = strtotime( $ymd . ' 00:00:00' );
        if ( ! $ts ) {
            return $ymd;
        }
        // Use WP timezone/locale.
        if ( function_exists( 'wp_date' ) ) {
            return wp_date( $this->php_date_format(), $ts );
        }
        return date_i18n( $this->php_date_format(), $ts );
    }

    /**
     * Format a MySQL datetime (YYYY-MM-DD HH:MM:SS) using the selected date format
     * (or WordPress date format) plus the WordPress time format.
     */
    private function display_datetime( $mysql_datetime ) {
        $mysql_datetime = is_string( $mysql_datetime ) ? $mysql_datetime : '';
        if ( $mysql_datetime === '' ) {
            return '';
        }

        // Expected format from current_time( 'mysql' ).
        $ts = 0;
        if ( function_exists( 'wp_timezone' ) ) {
            $tz = wp_timezone();
            $dt = date_create_from_format( 'Y-m-d H:i:s', $mysql_datetime, $tz );
            if ( $dt instanceof DateTime ) {
                $ts = (int) $dt->getTimestamp();
            }
        }
        if ( ! $ts ) {
            $ts = (int) strtotime( $mysql_datetime );
        }
        if ( ! $ts ) {
            // Fall back: show raw if we cannot parse.
            return $mysql_datetime;
        }

        $date_fmt = $this->php_date_format();
        $time_fmt = (string) get_option( 'time_format', 'H:i' );
        $fmt      = trim( $date_fmt . ' ' . $time_fmt );

        if ( function_exists( 'wp_date' ) ) {
            return wp_date( $fmt, $ts, function_exists( 'wp_timezone' ) ? wp_timezone() : null );
        }
        return date_i18n( $fmt, $ts );
    }



function valid_date( $date ) {
        $date = sanitize_text_field( (string) $date );
        if ( $date === '' ) {
            return '';
        }

        // Stored format is always YYYY-MM-DD.
        if ( preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
            return $date;
        }

        $key = $this->effective_date_format_key();
        $formats = array();

        if ( $key === 'mdy' ) {
            $formats = array( 'm/d/Y', 'm-d-Y' );
        } elseif ( $key === 'ymd' ) {
            $formats = array( 'Y-m-d', 'Y/m/d' );
        } else {
            $formats = array( 'd/m/Y', 'd-m-Y' );
        }

        foreach ( $formats as $fmt ) {
            $dt = DateTime::createFromFormat( $fmt, $date );
            if ( $dt instanceof DateTime ) {
                $errors = DateTime::getLastErrors();
                if ( empty( $errors['warning_count'] ) && empty( $errors['error_count'] ) ) {
                    return $dt->format( 'Y-m-d' );
                }
            }
        }

        return '';
    }

    private function next_due( $due, $recur ) {
        if ( $recur === 'none' ) {
            return $due;
        }
        if ( $due === '' ) {
            $due = $this->today();
        }

        try {
            $dt = new DateTime( $due );
        } catch ( Exception $e ) {
            return $due;
        }

        switch ( $recur ) {
            case 'daily':
                $dt->modify( '+1 day' );
                break;
            case 'weekly':
                $dt->modify( '+1 week' );
                break;
            case 'monthly':
                $dt->modify( '+1 month' );
                break;
            case 'quarterly':
                $dt->modify( '+3 months' );
                break;
        }

        return $dt->format( 'Y-m-d' );
    }

    private function calc_due_offset_days( $due ) {
        $due = $this->valid_date( $due );
        if ( $due === '' ) {
            return '';
        }
        try {
            $today = new DateTime( $this->today() );
            $dt    = new DateTime( $due );
            $diff  = (int) $today->diff( $dt )->format( '%r%a' );
            return (string) $diff;
        } catch ( Exception $e ) {
            return '';
        }
    }

    private function apply_due_offset_days( $offset ) {
        $offset = (string) $offset;
        if ( $offset === '' ) {
            return '';
        }
        $n = (int) $offset;
        try {
            $dt = new DateTime( $this->today() );
            $dt->modify( ( $n >= 0 ? '+' : '' ) . $n . ' days' );
            return $dt->format( 'Y-m-d' );
        } catch ( Exception $e ) {
            return '';
        }
    }

    /* -----------------------------
     * Handlers
     * ----------------------------- */

    // phpcs:disable WordPress.Security.NonceVerification.Missing
    public function handle_add() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_add' );

        $from_tab   = isset( $_POST['from_tab'] ) ? sanitize_key( wp_unslash( $_POST['from_tab'] ) ) : 'board';
        $return_tab = ( $from_tab === 'dashboard' ) ? 'board' : $from_tab;

        $title = isset( $_POST['task_title'] ) ? sanitize_text_field( wp_unslash( $_POST['task_title'] ) ) : '';
        $raw_due = isset( $_POST['task_due'] ) ? sanitize_text_field( wp_unslash( $_POST['task_due'] ) ) : '';
        $due     = $this->valid_date( $raw_due );
        $pri   = isset( $_POST['task_priority'] ) ? sanitize_key( wp_unslash( $_POST['task_priority'] ) ) : 'normal';
        $cat   = isset( $_POST['task_category'] ) ? sanitize_key( wp_unslash( $_POST['task_category'] ) ) : 'general';
        $recur = isset( $_POST['task_recur'] ) ? sanitize_key( wp_unslash( $_POST['task_recur'] ) ) : 'none';
        $notes = isset( $_POST['task_notes'] ) ? sanitize_text_field( wp_unslash( $_POST['task_notes'] ) ) : '';

	        $assignee = 0;
	        if ( $this->allow_assign() && current_user_can( 'list_users' ) ) {
	            $assignee = isset( $_POST['task_assignee'] ) ? absint( wp_unslash( $_POST['task_assignee'] ) ) : 0;
	            if ( $assignee && ! get_user_by( 'id', $assignee ) ) {
	                $assignee = 0;
	            }
	        }

        if ( $title === '' ) {
            wp_safe_redirect( $this->back_url( $return_tab ) );
            exit;
        }

        if ( ! in_array( $pri, array( 'low', 'normal', 'high' ), true ) ) {
            $pri = 'normal';
        }
        if ( ! in_array( $recur, array( 'none', 'daily', 'weekly', 'monthly', 'quarterly' ), true ) ) {
            $recur = 'none';
        }

        $cats = $this->get_categories();
        if ( ! isset( $cats[ $cat ] ) ) {
            $cat = 'general';
        }

        $tasks = $this->get_tasks();

        $max_sort = 0;
        foreach ( $tasks as $t ) {
            $s = isset( $t['sort'] ) ? (int) $t['sort'] : 0;
            if ( $s > $max_sort ) {
                $max_sort = $s;
            }
        }

        $id = $this->new_id();

        $tasks[ $id ] = array(
            'title'    => $title,
            'due'      => $due,
            'priority' => $pri,
            'category' => $cat,
            'recur'    => $recur,
            'assignee' => $assignee,
            'notes'    => $notes,
            'done'     => false,
            'created'  => current_time( 'mysql' ),
            'sort'     => $max_sort + 10,
        );

        $this->save_tasks( $tasks );
        $this->add_log( 'added', $title );

        wp_safe_redirect( $from_tab === 'dashboard' ? admin_url( 'index.php' ) : $this->back_url( $return_tab ) );
        exit;
    }

    public function handle_toggle() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_toggle' );

        $task_id    = isset( $_POST['task_id'] ) ? sanitize_text_field( wp_unslash( $_POST['task_id'] ) ) : '';
        $return_tab = isset( $_POST['return_tab'] ) ? sanitize_key( wp_unslash( $_POST['return_tab'] ) ) : 'board';

        $tasks = $this->get_tasks();
        if ( $task_id && isset( $tasks[ $task_id ] ) ) {
            $task  = $tasks[ $task_id ];
            $title = isset( $task['title'] ) ? (string) $task['title'] : '';

            $was_done    = ! empty( $task['done'] );
            $task['done'] = $was_done ? false : true;

            $recur = isset( $task['recur'] ) ? (string) $task['recur'] : 'none';
            $due   = isset( $task['due'] ) ? (string) $task['due'] : '';

            // If recurring and completing, mark this instance as completed
            // and create the next instance as a new open task.
            if ( ! $was_done && $task['done'] && $recur !== 'none' ) {

                // Complete the current task (so it moves to Completed tab)
                $task['done'] = true;

                // Create the next occurrence as a new task
                $next_id = $this->new_id();
                $next    = $task;
                $next['due']     = $this->next_due( $due, $recur );
                $next['done']    = false;
                $next['created'] = current_time( 'mysql' );

                // Put it at the end of the list
                $max_sort = 0;
                foreach ( $tasks as $t2 ) {
                    $s2 = isset( $t2['sort'] ) ? (int) $t2['sort'] : 0;
                    if ( $s2 > $max_sort ) {
                        $max_sort = $s2;
                    }
                }
                $next['sort'] = $max_sort + 10;

                $tasks[ $next_id ] = $next;

                $this->add_log( 'completed (recurring)', $title );
            } elseif ( ! $was_done && $task['done'] ) {
                $this->add_log( 'completed', $title );
            } else {
                $this->add_log( 'reopened', $title );
            }

            $tasks[ $task_id ] = $task;
            $this->save_tasks( $tasks );
        }

        wp_safe_redirect( $this->back_url( $return_tab ) );
        exit;
    }

    public function handle_delete() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_delete' );

        $task_id    = isset( $_POST['task_id'] ) ? sanitize_text_field( wp_unslash( $_POST['task_id'] ) ) : '';
        $return_tab = isset( $_POST['return_tab'] ) ? sanitize_key( wp_unslash( $_POST['return_tab'] ) ) : 'board';

        $tasks = $this->get_tasks();
        if ( $task_id && isset( $tasks[ $task_id ] ) ) {
            $title = isset( $tasks[ $task_id ]['title'] ) ? (string) $tasks[ $task_id ]['title'] : '';
            unset( $tasks[ $task_id ] );
            $this->save_tasks( $tasks );
            $this->add_log( 'deleted', $title );
        }

        wp_safe_redirect( $this->back_url( $return_tab ) );
        exit;
    }

    public function handle_bulk() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_bulk' );

        $action = isset( $_POST['bulk_action'] ) ? sanitize_key( wp_unslash( $_POST['bulk_action'] ) ) : '';
        $ids    = isset( $_POST['task_ids'] ) && is_array( $_POST['task_ids'] )
            ? array_map( 'sanitize_text_field', wp_unslash( $_POST['task_ids'] ) )
            : array();

        $return_tab = isset( $_POST['return_tab'] ) ? sanitize_key( wp_unslash( $_POST['return_tab'] ) ) : 'board';

        if ( $action === '' || empty( $ids ) ) {
            wp_safe_redirect( $this->back_url( $return_tab ) );
            exit;
        }

        $tasks = $this->get_tasks();

        
        if ( $action === 'save_template' ) {
            $tpl_name = isset( $_POST['bulk_tpl_name'] ) ? sanitize_text_field( wp_unslash( $_POST['bulk_tpl_name'] ) ) : '';
            $tpl_desc = isset( $_POST['bulk_tpl_desc'] ) ? sanitize_text_field( wp_unslash( $_POST['bulk_tpl_desc'] ) ) : '';

            $draft_items = array();

            foreach ( $ids as $id ) {
                if ( ! isset( $tasks[ $id ] ) ) {
                    continue;
                }
                $t = $tasks[ $id ];

                $draft_items[] = array(
                    'title'      => isset( $t['title'] ) ? sanitize_text_field( (string) $t['title'] ) : '',
                    'priority'   => isset( $t['priority'] ) ? sanitize_key( (string) $t['priority'] ) : 'normal',
                    'category'   => isset( $t['category'] ) ? sanitize_key( (string) $t['category'] ) : 'general',
                    'recur'      => isset( $t['recur'] ) ? sanitize_key( (string) $t['recur'] ) : 'none',
                    'notes'      => isset( $t['notes'] ) ? sanitize_text_field( (string) $t['notes'] ) : '',
                    'due_offset' => $this->calc_due_offset_days( isset( $t['due'] ) ? (string) $t['due'] : '' ),
                );
            }

            // If a name was provided, save immediately (best UX). Otherwise keep the draft flow.
            if ( $tpl_name !== '' && ! empty( $draft_items ) ) {
                $tpls  = $this->get_user_templates();
                $tpl_id = 'tpl_' . $this->new_id();

                $tpls[ $tpl_id ] = array(
                    'name'    => $tpl_name,
                    'desc'    => $tpl_desc,
                    'items'   => array_values( $draft_items ),
                    'created' => current_time( 'mysql' ),
                );

                $this->save_user_templates( $tpls );
                delete_transient( self::TRANSIENT_TEMPLATE_DRAFT . (int) get_current_user_id() );

                $this->add_log( 'template saved', $tpl_name );

                wp_safe_redirect( $this->back_url( 'templates' ) );
                exit;
            }

            set_transient(
                self::TRANSIENT_TEMPLATE_DRAFT . (int) get_current_user_id(),
                array( 'items' => $draft_items, 'created' => time() ),
                10 * MINUTE_IN_SECONDS
            );

            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        foreach ( $ids as $id ) {
            if ( ! isset( $tasks[ $id ] ) ) {
                continue;
            }

            $title = isset( $tasks[ $id ]['title'] ) ? (string) $tasks[ $id ]['title'] : '';

            if ( $action === 'done' ) {
                $tasks[ $id ]['done'] = true;
                $this->add_log( 'bulk completed', $title );
            } elseif ( $action === 'undone' ) {
                $tasks[ $id ]['done'] = false;
                $this->add_log( 'bulk reopened', $title );
            } elseif ( $action === 'delete' ) {
                unset( $tasks[ $id ] );
                $this->add_log( 'bulk deleted', $title );
            }
        }

        $this->save_tasks( $tasks );
        wp_safe_redirect( $this->back_url( $return_tab ) );
        exit;
    }

    public function handle_clear_completed() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_clear_completed' );

        $tasks = $this->get_tasks();
        foreach ( $tasks as $id => $t ) {
            if ( ! empty( $t['done'] ) ) {
                $title = isset( $t['title'] ) ? (string) $t['title'] : '';
                unset( $tasks[ $id ] );
                $this->add_log( 'cleared completed', $title );
            }
        }
        $this->save_tasks( $tasks );

        wp_safe_redirect( $this->back_url( 'completed' ) );
        exit;
    }

    public function handle_add_builtin_template() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_add_template' );

        $key = isset( $_POST['template_key'] ) ? sanitize_key( wp_unslash( $_POST['template_key'] ) ) : '';
        $templates = $this->templates();
        if ( ! isset( $templates[ $key ] ) ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        $set   = $templates[ $key ]['tasks'];
        $tasks = $this->get_tasks();

        $max_sort = 0;
        foreach ( $tasks as $t ) {
            $s = isset( $t['sort'] ) ? (int) $t['sort'] : 0;
            if ( $s > $max_sort ) {
                $max_sort = $s;
            }
        }

        foreach ( $set as $tpl_task ) {
            $id = $this->new_id();
            $max_sort += 10;

            $tasks[ $id ] = array(
                'title'    => sanitize_text_field( (string) $tpl_task['title'] ),
                'due'      => $this->valid_date( (string) $tpl_task['due'] ),
                'priority' => sanitize_key( (string) $tpl_task['priority'] ),
                'category' => sanitize_key( (string) $tpl_task['category'] ),
                'recur'    => sanitize_key( (string) $tpl_task['recur'] ),
                'assignee' => 0,
                'notes'    => sanitize_text_field( (string) $tpl_task['notes'] ),
                'done'     => false,
                'created'  => current_time( 'mysql' ),
                'sort'     => $max_sort,
            );

            $this->add_log( 'template added', (string) $tpl_task['title'] );
        }

        $this->save_tasks( $tasks );
        wp_safe_redirect( $this->back_url( 'board' ) );
        exit;
    }

    public function handle_prepare_template_from_tasks() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_prepare_template_from_tasks' );
        wp_safe_redirect( $this->back_url( 'board' ) );
        exit;
    }

    public function handle_create_user_template() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_create_user_template' );

        $name = isset( $_POST['tpl_name'] ) ? sanitize_text_field( wp_unslash( $_POST['tpl_name'] ) ) : '';
        $desc = isset( $_POST['tpl_desc'] ) ? sanitize_text_field( wp_unslash( $_POST['tpl_desc'] ) ) : '';

        $draft = get_transient( self::TRANSIENT_TEMPLATE_DRAFT . (int) get_current_user_id() );
        if ( $name === '' || ! is_array( $draft ) || empty( $draft['items'] ) ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        $tpls = $this->get_user_templates();
        $tpl_id = 'tpl_' . $this->new_id();

        $tpls[ $tpl_id ] = array(
            'name'    => $name,
            'desc'    => $desc,
            'items'   => array_values( $draft['items'] ),
            'created' => current_time( 'mysql' ),
        );

        $this->save_user_templates( $tpls );
        delete_transient( self::TRANSIENT_TEMPLATE_DRAFT . (int) get_current_user_id() );

        $this->add_log( 'template saved', $name );

        wp_safe_redirect( $this->back_url( 'templates' ) );
        exit;
    }

    public function handle_apply_user_template() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_apply_user_template' );

        $tpl_id = isset( $_POST['tpl_id'] ) ? sanitize_text_field( wp_unslash( $_POST['tpl_id'] ) ) : '';
        $tpls   = $this->get_user_templates();

        if ( ! $tpl_id || ! isset( $tpls[ $tpl_id ] ) ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        $tpl = $tpls[ $tpl_id ];
        $items = isset( $tpl['items'] ) && is_array( $tpl['items'] ) ? $tpl['items'] : array();
        if ( empty( $items ) ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        $tasks = $this->get_tasks();

        $max_sort = 0;
        foreach ( $tasks as $t ) {
            $s = isset( $t['sort'] ) ? (int) $t['sort'] : 0;
            if ( $s > $max_sort ) {
                $max_sort = $s;
            }
        }

        foreach ( $items as $it ) {
            $id = $this->new_id();
            $max_sort += 10;

            $tasks[ $id ] = array(
                'title'    => isset( $it['title'] ) ? sanitize_text_field( (string) $it['title'] ) : '',
                'due'      => $this->apply_due_offset_days( isset( $it['due_offset'] ) ? (string) $it['due_offset'] : '' ),
                'priority' => isset( $it['priority'] ) ? sanitize_key( (string) $it['priority'] ) : 'normal',
                'category' => isset( $it['category'] ) ? sanitize_key( (string) $it['category'] ) : 'general',
                'recur'    => isset( $it['recur'] ) ? sanitize_key( (string) $it['recur'] ) : 'none',
                'assignee' => 0,
                'notes'    => isset( $it['notes'] ) ? sanitize_text_field( (string) $it['notes'] ) : '',
                'done'     => false,
                'created'  => current_time( 'mysql' ),
                'sort'     => $max_sort,
            );
        }

        $this->save_tasks( $tasks );
        $tpl_name = isset( $tpl['name'] ) ? (string) $tpl['name'] : '';
        $this->add_log( 'template applied', $tpl_name );

        wp_safe_redirect( $this->back_url( 'board' ) );
        exit;
    }

    public function handle_delete_user_template() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_post_nonce( 'of_tb_delete_user_template' );

        $tpl_id = isset( $_POST['tpl_id'] ) ? sanitize_text_field( wp_unslash( $_POST['tpl_id'] ) ) : '';
        $tpls   = $this->get_user_templates();

        if ( $tpl_id && isset( $tpls[ $tpl_id ] ) ) {
            $name = isset( $tpls[ $tpl_id ]['name'] ) ? (string) $tpls[ $tpl_id ]['name'] : '';
            unset( $tpls[ $tpl_id ] );
            $this->save_user_templates( $tpls );
            $this->add_log( 'template deleted', $name );
        }

        wp_safe_redirect( $this->back_url( 'templates' ) );
        exit;
    }

    /* -----------------------------
     * AJAX reorder
     * ----------------------------- */

    public function ajax_reorder() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_send_json_error( 'access_denied' );
        }

        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, 'of_tb_reorder' ) ) {
            wp_send_json_error( 'bad_nonce' );
        }

        $order = isset( $_POST['order'] ) && is_array( $_POST['order'] )
            ? array_map( 'sanitize_text_field', wp_unslash( $_POST['order'] ) )
            : array();

        if ( empty( $order ) ) {
            wp_send_json_success();
        }

        $tasks = $this->get_tasks();

        $sort = 10;
        foreach ( $order as $id ) {
            if ( isset( $tasks[ $id ] ) ) {
                $tasks[ $id ]['sort'] = $sort;
                $sort += 10;
            }
        }

        $this->save_tasks( $tasks );
        wp_send_json_success();
    }

    /* -----------------------------
     * CSV export (tasks + log + templates)
     * ----------------------------- */

    public function handle_export_tasks_csv() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_get_nonce( 'of_tb_export_tasks' );

        $tasks = $this->get_tasks();
        $cats  = $this->get_categories();

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=otterfixer-task-board-tasks.csv' );

        $out = fopen( 'php://output', 'w' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        fputcsv( $out, array( 'Title', 'Done', 'Due', 'Priority', 'Category', 'Repeats', 'Assignee', 'Notes', 'Created' ) );

        foreach ( $tasks as $t ) {
            $cat = isset( $t['category'] ) ? (string) $t['category'] : 'general';
            $cat_label = isset( $cats[ $cat ]['label'] ) ? (string) $cats[ $cat ]['label'] : $cat;

            $assignee_name = '';
            $assignee = isset( $t['assignee'] ) ? (int) $t['assignee'] : 0;
            if ( $assignee > 0 ) {
                $u = get_user_by( 'id', $assignee );
                if ( $u ) {
                    $assignee_name = $u->display_name;
                }
            }

            fputcsv( $out, array(
                isset( $t['title'] ) ? (string) $t['title'] : '',
                ! empty( $t['done'] ) ? 'yes' : 'no',
                isset( $t['due'] ) ? (string) $t['due'] : '',
                isset( $t['priority'] ) ? (string) $t['priority'] : '',
                $cat_label,
                isset( $t['recur'] ) ? (string) $t['recur'] : 'none',
                $assignee_name,
                isset( $t['notes'] ) ? (string) $t['notes'] : '',
                isset( $t['created'] ) ? (string) $t['created'] : '',
            ) );
        }

        fclose( $out ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        exit;
    }

    public function handle_export_log_csv() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_get_nonce( 'of_tb_export_log' );

        $log = $this->get_log();

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=otterfixer-task-board-log.csv' );

        $out = fopen( 'php://output', 'w' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        fputcsv( $out, array( 'When', 'User', 'Action', 'Task' ) );

        foreach ( $log as $row ) {
            fputcsv( $out, array(
                isset( $row['when'] ) ? (string) $row['when'] : '',
                isset( $row['user'] ) ? (string) $row['user'] : '',
                isset( $row['action'] ) ? (string) $row['action'] : '',
                isset( $row['task'] ) ? (string) $row['task'] : '',
            ) );
        }

        fclose( $out ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        exit;
    }

    public function handle_export_templates_csv() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_get_nonce( 'of_tb_export_templates' );

        $tpls = $this->get_user_templates();

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=otterfixer-task-board-templates.csv' );

        $out = fopen( 'php://output', 'w' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        fputcsv( $out, array( 'Template Name', 'Template Description', 'Task Title', 'Due Offset Days', 'Priority', 'Category', 'Repeats', 'Notes' ) );

        foreach ( $tpls as $tpl ) {
            $name = isset( $tpl['name'] ) ? (string) $tpl['name'] : '';
            $desc = isset( $tpl['desc'] ) ? (string) $tpl['desc'] : '';
            $items = isset( $tpl['items'] ) && is_array( $tpl['items'] ) ? $tpl['items'] : array();

            if ( empty( $items ) ) {
                fputcsv( $out, array( $name, $desc, '', '', '', '', '', '' ) );
                continue;
            }

            foreach ( $items as $it ) {
                fputcsv( $out, array(
                    $name,
                    $desc,
                    isset( $it['title'] ) ? (string) $it['title'] : '',
                    isset( $it['due_offset'] ) ? (string) $it['due_offset'] : '',
                    isset( $it['priority'] ) ? (string) $it['priority'] : '',
                    isset( $it['category'] ) ? (string) $it['category'] : '',
                    isset( $it['recur'] ) ? (string) $it['recur'] : '',
                    isset( $it['notes'] ) ? (string) $it['notes'] : '',
                ) );
            }
        }

        fclose( $out ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        exit;
    }

    public function handle_download_templates_template_csv() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }
        $this->verify_get_nonce( 'of_tb_download_templates_template' );

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=otterfixer-task-board-templates-template.csv' );

        $out = fopen( 'php://output', 'w' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen

        // Matches the plugin's exported format exactly.
        fputcsv( $out, array( 'Template Name', 'Template Description', 'Task Title', 'Due Offset Days', 'Priority', 'Category', 'Repeats', 'Notes' ) );

        // Example template with two tasks.
        fputcsv( $out, array( 'My Template', 'Optional description', 'Check SSL certificate', '0', 'normal', 'security', 'none', '' ) );
        fputcsv( $out, array( 'My Template', 'Optional description', 'Run WordPress updates', '0', 'normal', 'updates', 'monthly', 'Do a backup first' ) );

        fclose( $out ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        exit;
    }

    public function handle_import_templates_csv() {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Access denied.', 'otterfixer-task-board' ) );
        }

        $this->verify_post_nonce( 'of_tb_import_templates' );

        if ( empty( $_FILES['tpl_csv'] ) || ! is_array( $_FILES['tpl_csv'] ) ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $tmp_name = isset( $_FILES['tpl_csv']['tmp_name'] ) ? (string) $_FILES['tpl_csv']['tmp_name'] : '';
        $tmp_name = wp_normalize_path( $tmp_name );

        if ( $tmp_name === '' || ! file_exists( $tmp_name ) || ! is_uploaded_file( $tmp_name ) ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        // Read via WP_Filesystem to satisfy plugin checks.
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        global $wp_filesystem;
        WP_Filesystem();

        if ( ! $wp_filesystem || ! method_exists( $wp_filesystem, 'get_contents' ) ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        $contents = (string) $wp_filesystem->get_contents( $tmp_name );
        if ( $contents === '' ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        // Parse CSV from memory.
        $stream = fopen( 'php://temp', 'r+' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
        if ( ! $stream ) {
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        fwrite( $stream, $contents ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite
        rewind( $stream );

        $header = fgetcsv( $stream );
        if ( ! is_array( $header ) || empty( $header ) ) {
            fclose( $stream ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
            wp_safe_redirect( $this->back_url( 'templates' ) );
            exit;
        }

        // Map columns based on exported format:
        // Template Name, Template Description, Task Title, Due Offset Days, Priority, Category, Repeats, Notes
        $col = array();
        foreach ( $header as $i => $h ) {
            $k = strtolower( trim( (string) $h ) );
            $col[ $k ] = (int) $i;
        }

        $idx_name = isset( $col['template name'] ) ? $col['template name'] : 0;
        $idx_desc = isset( $col['template description'] ) ? $col['template description'] : 1;
        $idx_task = isset( $col['task title'] ) ? $col['task title'] : 2;
        $idx_due  = isset( $col['due offset days'] ) ? $col['due offset days'] : 3;
        $idx_pri  = isset( $col['priority'] ) ? $col['priority'] : 4;
        $idx_cat  = isset( $col['category'] ) ? $col['category'] : 5;
        $idx_rep  = isset( $col['repeats'] ) ? $col['repeats'] : 6;
        $idx_not  = isset( $col['notes'] ) ? $col['notes'] : 7;

        $existing = $this->get_user_templates();

        // Collect existing names so imports don't overwrite.
        $existing_names = array();
        foreach ( $existing as $t ) {
            $nm = isset( $t['name'] ) ? strtolower( (string) $t['name'] ) : '';
            if ( $nm !== '' ) {
                $existing_names[ $nm ] = true;
            }
        }

        // Group rows into templates by name+desc.
        $groups = array();

        while ( ( $row = fgetcsv( $stream ) ) !== false ) {
            if ( ! is_array( $row ) || empty( $row ) ) {
                continue;
            }

            $name = isset( $row[ $idx_name ] ) ? sanitize_text_field( (string) $row[ $idx_name ] ) : '';
            $desc = isset( $row[ $idx_desc ] ) ? sanitize_text_field( (string) $row[ $idx_desc ] ) : '';

            if ( $name === '' ) {
                continue;
            }

            $gkey = strtolower( $name . '||' . $desc );

            if ( ! isset( $groups[ $gkey ] ) ) {
                $groups[ $gkey ] = array(
                    'name'  => $name,
                    'desc'  => $desc,
                    'items' => array(),
                );
            }

            $task_title = isset( $row[ $idx_task ] ) ? sanitize_text_field( (string) $row[ $idx_task ] ) : '';
            if ( $task_title === '' ) {
                continue; // allow template shell, but no task on this row
            }

            $due_raw = isset( $row[ $idx_due ] ) ? trim( (string) $row[ $idx_due ] ) : '';
            $due_raw = preg_replace( '/[^0-9\-]/', '', (string) $due_raw );
            $due_offset = ( $due_raw !== '' ) ? (string) (int) $due_raw : '';

            $priority = isset( $row[ $idx_pri ] ) ? sanitize_key( (string) $row[ $idx_pri ] ) : 'normal';
            $category = isset( $row[ $idx_cat ] ) ? sanitize_key( (string) $row[ $idx_cat ] ) : 'general';
            $recur    = isset( $row[ $idx_rep ] ) ? sanitize_key( (string) $row[ $idx_rep ] ) : 'none';
            $notes    = isset( $row[ $idx_not ] ) ? sanitize_text_field( (string) $row[ $idx_not ] ) : '';

            $groups[ $gkey ]['items'][] = array(
                'title'      => $task_title,
                'priority'   => $priority !== '' ? $priority : 'normal',
                'category'   => $category !== '' ? $category : 'general',
                'recur'      => $recur !== '' ? $recur : 'none',
                'notes'      => $notes,
                'due_offset' => $due_offset,
            );
        }

        fclose( $stream ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose

        // Merge groups into existing templates, without overwriting.
        $imported_count = 0;

        foreach ( $groups as $g ) {
            $name = isset( $g['name'] ) ? (string) $g['name'] : '';
            $desc = isset( $g['desc'] ) ? (string) $g['desc'] : '';
            $items = isset( $g['items'] ) && is_array( $g['items'] ) ? $g['items'] : array();

            if ( $name === '' ) {
                continue;
            }

            $base = $name;
            $candidate = $base;
            $n = 2;
            while ( isset( $existing_names[ strtolower( $candidate ) ] ) ) {
                $candidate = $base . ' (' . (string) $n . ')';
                $n++;
            }
            $existing_names[ strtolower( $candidate ) ] = true;

            $tpl_id = 'tpl_' . $this->new_id();

            $existing[ $tpl_id ] = array(
                'name'    => $candidate,
                'desc'    => $desc,
                'items'   => array_values( $items ),
                'created' => current_time( 'mysql' ),
            );

            $imported_count++;
        }

        if ( count( $existing ) > 60 ) {
            $existing = array_slice( $existing, -60, null, true );
        }

        $this->save_user_templates( $existing );

        if ( $imported_count > 0 ) {
            $this->add_log( 'templates imported', 'CSV (' . (string) $imported_count . ')' );
        } else {
            $this->add_log( 'templates import', 'CSV (no valid rows)' );
        }

        wp_safe_redirect( $this->back_url( 'templates' ) );
        exit;
    }
    // phpcs:enable WordPress.Security.NonceVerification.Missing

    /* -----------------------------
     * Templates (built in)
     * ----------------------------- */

    private function templates() {
        $today = $this->today();

        return array(
            'maintenance_monthly' => array(
                'name' => __( 'Monthly Website Checks', 'otterfixer-task-board' ),
                'desc' => __( 'A sensible monthly routine for updates, backups, security, and SEO checks.', 'otterfixer-task-board' ),
                'tasks' => array(
                    array(
                        'title' => 'Check plugin updates (test if possible)',
                        'due' => $today,
                        'priority' => 'normal',
                        'category' => 'updates',
                        'recur' => 'monthly',
                        'notes' => 'If it is a major update, consider staging first.',
                    ),
                    array(
                        'title' => 'Confirm backups are running and restorable',
                        'due' => $today,
                        'priority' => 'high',
                        'category' => 'hosting',
                        'recur' => 'monthly',
                        'notes' => 'A backup is only useful if you can restore it.',
                    ),
                    array(
                        'title' => 'Review user accounts and admin access',
                        'due' => $today,
                        'priority' => 'high',
                        'category' => 'security',
                        'recur' => 'monthly',
                        'notes' => 'Remove unused accounts and verify roles.',
                    ),
                    array(
                        'title' => 'Quick SEO review (broken links and key pages)',
                        'due' => $today,
                        'priority' => 'normal',
                        'category' => 'seo',
                        'recur' => 'monthly',
                        'notes' => 'Spot check indexing in Search Console if available.',
                    ),
                ),
            ),

            'launch_checklist' => array(
                'name' => __( 'New Site Launch Checklist', 'otterfixer-task-board' ),
                'desc' => __( 'A practical checklist for launching a WordPress site without missing the basics.', 'otterfixer-task-board' ),
                'tasks' => array(
                    array(
                        'title' => 'Set site title, tagline, and permalinks',
                        'due' => $today,
                        'priority' => 'normal',
                        'category' => 'general',
                        'recur' => 'none',
                        'notes' => 'Confirm permalink structure matches your plan.',
                    ),
                    array(
                        'title' => 'Install analytics and confirm tracking',
                        'due' => $today,
                        'priority' => 'normal',
                        'category' => 'seo',
                        'recur' => 'none',
                        'notes' => 'Test key events or conversions if you track them.',
                    ),
                    array(
                        'title' => 'Set up backups and security settings',
                        'due' => $today,
                        'priority' => 'high',
                        'category' => 'security',
                        'recur' => 'none',
                        'notes' => 'Make sure you can restore quickly if needed.',
                    ),
                    array(
                        'title' => 'Cache setup and performance sanity check',
                        'due' => $today,
                        'priority' => 'normal',
                        'category' => 'hosting',
                        'recur' => 'none',
                        'notes' => 'Avoid aggressive caching until stable.',
                    ),
                ),
            ),

            'security_basics' => array(
                'name' => __( 'Security Basics', 'otterfixer-task-board' ),
                'desc' => __( 'Good foundations: reduce risk without overcomplicating things.', 'otterfixer-task-board' ),
                'tasks' => array(
                    array(
                        'title' => 'Enable 2FA for admins (if possible)',
                        'due' => $today,
                        'priority' => 'high',
                        'category' => 'security',
                        'recur' => 'none',
                        'notes' => 'Start with administrator accounts.',
                    ),
                    array(
                        'title' => 'Remove unused plugins and themes',
                        'due' => $today,
                        'priority' => 'high',
                        'category' => 'security',
                        'recur' => 'none',
                        'notes' => 'Less surface area means fewer problems.',
                    ),
                    array(
                        'title' => 'Quarterly permissions check',
                        'due' => $today,
                        'priority' => 'normal',
                        'category' => 'hosting',
                        'recur' => 'quarterly',
                        'notes' => 'Keep changes minimal and stable.',
                    ),
                ),
            ),
        );
    }
}

new OtterFixer_Task_Board();