<?php
/*
Plugin Name: OtterFixer Simple AI Meta Descriptions Generator
Description: Generate human-friendly meta descriptions for posts and pages using the OpenAI API. Includes per-post generate buttons, a bulk generator, and safe sentence-ending trimming.
Version: 1.0.5
Author: OtterFixer
License: GPLv2 or later
Text Domain: otterfixer-simple-ai-meta-descriptions-generator
Domain Path: /languages
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

add_action( 'plugins_loaded', function() {
    load_plugin_textdomain( 'otterfixer-simple-ai-meta-descriptions-generator', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
} );


final class OFAI_Meta_Descriptions {

    const OPT_API_KEY          = 'ofai_openai_api_key';
    const OPT_MODEL            = 'ofai_openai_model';
    const OPT_CHAR_LIMIT       = 'ofai_char_limit';
    const OPT_MAX_WORDS        = 'ofai_max_words';
    const OPT_TONE             = 'ofai_tone';
    const OPT_GEN_ON_PUBLISH   = 'ofai_gen_on_publish';
    const OPT_ALLOWED_TYPES    = 'ofai_allowed_post_types';

    const META_DESC            = '_ofai_meta_description';
    const META_LOCK            = '_ofai_meta_lock';
    const META_UPDATED_AT      = '_ofai_meta_updated_at';
    const META_UPDATED_BY      = '_ofai_meta_updated_by';

    const NONCE_AJAX           = 'ofai_ajax_nonce';
    const NONCE_SETTINGS       = 'ofai_settings_nonce';
    const NONCE_BULK           = 'ofai_bulk_nonce';

    const BULK_TRANSIENT_PREFIX = 'ofai_bulk_';

    public static function init() : void {
        add_action( 'admin_menu', array( __CLASS__, 'admin_menu' ) );
        add_action( 'admin_init', array( __CLASS__, 'register_settings' ) );

        add_action( 'add_meta_boxes', array( __CLASS__, 'add_meta_box' ) );
        add_action( 'save_post', array( __CLASS__, 'save_post_meta' ), 10, 2 );

        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_assets' ) );

        add_action( 'wp_head', array( __CLASS__, 'output_meta_description' ), 1 );

        add_action( 'wp_ajax_ofai_generate', array( __CLASS__, 'ajax_generate' ) );
        add_action( 'wp_ajax_ofai_bulk_init', array( __CLASS__, 'ajax_bulk_init' ) );
        add_action( 'wp_ajax_ofai_bulk_process', array( __CLASS__, 'ajax_bulk_process' ) );
    }

    public static function required_cap() : string {
        return 'edit_posts';
    }

    public static function can_use() : bool {
        return current_user_can( self::required_cap() );
    }

    public static function admin_menu() : void {
        add_options_page(
            'AI Meta Descriptions',
            'AI Meta Descriptions',
            'manage_options',
            'ofai-meta-settings',
            array( __CLASS__, 'settings_page' )
        );

        add_management_page(
            'AI Meta Descriptions (Bulk)',
            'AI Meta Descriptions',
            self::required_cap(),
            'ofai-meta-bulk',
            array( __CLASS__, 'bulk_page' )
        );
    }

    public static function register_settings() : void {
        register_setting( 'ofai_settings', self::OPT_API_KEY, array(
            'type'              => 'string',
            'sanitize_callback' => array( __CLASS__, 'sanitize_api_key' ),
            'default'           => '',
        ) );

        register_setting( 'ofai_settings', self::OPT_MODEL, array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => 'gpt-4o-mini',
        ) );

        register_setting( 'ofai_settings', self::OPT_CHAR_LIMIT, array(
            'type'              => 'integer',
            'sanitize_callback' => array( __CLASS__, 'sanitize_int_range' ),
            'default'           => 155,
        ) );

        register_setting( 'ofai_settings', self::OPT_MAX_WORDS, array(
            'type'              => 'integer',
            'sanitize_callback' => array( __CLASS__, 'sanitize_int_range' ),
            'default'           => 800,
        ) );

        register_setting( 'ofai_settings', self::OPT_TONE, array(
            'type'              => 'string',
            'sanitize_callback' => array( __CLASS__, 'sanitize_tone' ),
            'default'           => 'neutral',
        ) );

        register_setting( 'ofai_settings', self::OPT_GEN_ON_PUBLISH, array(
            'type'              => 'string',
            'sanitize_callback' => array( __CLASS__, 'sanitize_onoff' ),
            'default'           => 'off',
        ) );

        register_setting( 'ofai_settings', self::OPT_ALLOWED_TYPES, array(
            'type'              => 'array',
            'sanitize_callback' => array( __CLASS__, 'sanitize_post_types' ),
            'default'           => array( 'post', 'page' ),
        ) );
    }

    public static function sanitize_api_key( $value ) : string {
        $value = is_string( $value ) ? trim( $value ) : '';
        // Allow typical key formats, but do not over-restrict.
        $value = preg_replace( '/\s+/', '', $value );
        return sanitize_text_field( $value );
    }

    public static function sanitize_int_range( $value ) : int {
        $v = intval( $value );
        if ( $v < 50 )  $v = 50;
        if ( $v > 2000 ) $v = 2000;
        return $v;
    }

    public static function sanitize_tone( $value ) : string {
        $value = is_string( $value ) ? strtolower( trim( $value ) ) : 'neutral';
        $allowed = array( 'neutral', 'professional', 'friendly' );
        return in_array( $value, $allowed, true ) ? $value : 'neutral';
    }

    public static function sanitize_onoff( $value ) : string {
        $value = is_string( $value ) ? strtolower( trim( $value ) ) : 'off';
        return ( $value === 'on' ) ? 'on' : 'off';
    }

    public static function sanitize_post_types( $value ) : array {
        if ( ! is_array( $value ) ) {
            return array( 'post', 'page' );
        }
        $out = array();
        foreach ( $value as $pt ) {
            $pt = sanitize_key( $pt );
            if ( $pt ) {
                $out[] = $pt;
            }
        }
        $out = array_values( array_unique( $out ) );
        return $out ? $out : array( 'post', 'page' );
    }

    public static function get_option_str( string $key, string $default = '' ) : string {
        $v = get_option( $key, $default );
        return is_string( $v ) ? $v : $default;
    }

    public static function get_option_int( string $key, int $default ) : int {
        $v = get_option( $key, $default );
        return intval( $v );
    }

    public static function get_allowed_post_types() : array {
        $types = get_option( self::OPT_ALLOWED_TYPES, array( 'post', 'page' ) );
        return is_array( $types ) ? $types : array( 'post', 'page' );
    }

    public static function enqueue_assets( string $hook ) : void {
        if ( ! self::can_use() ) {
            return;
        }

        if ( in_array( $hook, array( 'post.php', 'post-new.php', 'tools_page_ofai-meta-bulk' ), true ) ) {
            wp_enqueue_style(
                'ofai-admin',
                plugins_url( 'assets/ofai-admin.css', __FILE__ ),
                array(),
                '1.0.2');

            wp_enqueue_script(
                'ofai-admin',
                plugins_url( 'assets/ofai-admin.js', __FILE__ ),
                array( 'jquery' ),
                '1.0.2',
                true
            );

            wp_localize_script(
                'ofai-admin',
                'OFAI',
                array(
                    'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                    'nonce'   => wp_create_nonce( self::NONCE_AJAX ),
                    'i18n'    => array(
                        'generating' => 'Generating...',
                        'saving'     => 'Saving...',
                        'done'       => 'Done',
                        'nothingToDo' => 'Nothing to do. All meta descriptions are already generated.',
                        'error'      => 'Something went wrong. Please try again.',
                        'noKey'      => 'OpenAI API key is missing. Please add it in Settings.',
                    ),
                )
            );
        }
    }

    public static function add_meta_box() : void {
        $types = self::get_allowed_post_types();
        foreach ( $types as $pt ) {
            add_meta_box(
                'ofai_meta_box',
                'AI Meta Description',
                array( __CLASS__, 'render_meta_box' ),
                $pt,
                'side',
                'high'
            );
        }
    }

    public static function render_meta_box( WP_Post $post ) : void {
        if ( ! self::can_use() ) {
            return;
        }

        $desc   = get_post_meta( $post->ID, self::META_DESC, true );
        $lock   = get_post_meta( $post->ID, self::META_LOCK, true );
        $lock   = ( $lock === '1' ) ? '1' : '0';

        $updated_at = intval( get_post_meta( $post->ID, self::META_UPDATED_AT, true ) );
        $updated_by = intval( get_post_meta( $post->ID, self::META_UPDATED_BY, true ) );

        $by_name = '';
        if ( $updated_by ) {
            $u = get_user_by( 'id', $updated_by );
            if ( $u && ! is_wp_error( $u ) ) {
                $by_name = $u->display_name;
            }
        }

        $updated_str = '';
        if ( $updated_at ) {
            $updated_str = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $updated_at );
        }

        wp_nonce_field( 'ofai_save_meta', 'ofai_save_meta_nonce' );
        ?>
        <p>
            <label for="ofai_meta_desc"><strong>Meta description</strong></label>
        </p>
        <textarea id="ofai_meta_desc" name="ofai_meta_desc" class="ofai-textarea" rows="4" placeholder="Generate or write a meta description."><?php echo esc_textarea( (string) $desc ); ?></textarea>
        <p class="ofai-small">
            <span class="ofai-count" data-limit="<?php echo esc_attr( self::get_option_int( self::OPT_CHAR_LIMIT, 155 ) ); ?>"></span>
        </p>

        <p>
            <label>
                <input type="checkbox" name="ofai_meta_lock" value="1" <?php checked( $lock, '1' ); ?> />
                Lock this description
            </label>
        </p>

        <p class="ofai-actions">
            <button type="button" class="button button-secondary ofai-generate" data-post-id="<?php echo esc_attr( $post->ID ); ?>" data-mode="generate">Generate</button>
            <button type="button" class="button ofai-regenerate" data-post-id="<?php echo esc_attr( $post->ID ); ?>" data-mode="regenerate">Regenerate</button>
        </p>

        <p class="ofai-status" aria-live="polite"></p>

<div class="ofai-preview">
    <strong>SEO preview</strong>
    <div class="ofai-preview-box" role="note" aria-label="SEO preview">
        <div class="ofai-preview-title"><?php echo esc_html( get_the_title( $post ) ); ?></div>
        <div class="ofai-preview-url"><?php echo esc_html( get_permalink( $post ) ); ?></div>
        <div class="ofai-preview-desc"><?php echo esc_html( $desc ? $desc : 'Meta description preview will appear here.' ); ?></div>
    </div>
</div>


        <?php if ( $updated_str || $by_name ) : ?>
            <p class="ofai-small">
                <?php if ( $updated_str ) : ?>
                    Last updated: <?php echo esc_html( $updated_str ); ?><br />
                <?php endif; ?>
                <?php if ( $by_name ) : ?>
                    Updated by: <?php echo esc_html( $by_name ); ?>
                <?php endif; ?>
            </p>
        <?php endif; ?>
        <?php
    }

    public static function save_post_meta( int $post_id, WP_Post $post ) : void {
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( ! isset( $_POST['ofai_save_meta_nonce'] ) ) {
            return;
        }

        $nonce = sanitize_text_field( wp_unslash( $_POST['ofai_save_meta_nonce'] ) );
        if ( ! wp_verify_nonce( $nonce, 'ofai_save_meta' ) ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Save lock.
        $lock = isset( $_POST['ofai_meta_lock'] ) ? '1' : '0';
        update_post_meta( $post_id, self::META_LOCK, $lock );

        // Save manual edits.
        if ( isset( $_POST['ofai_meta_desc'] ) ) {
            $desc_raw = wp_unslash( $_POST['ofai_meta_desc'] );
            $desc     = sanitize_textarea_field( $desc_raw );

            $limit = self::get_option_int( self::OPT_CHAR_LIMIT, 155 );
            $desc  = self::format_meta( $desc, $limit );

            update_post_meta( $post_id, self::META_DESC, $desc );
            update_post_meta( $post_id, self::META_UPDATED_AT, time() );
            update_post_meta( $post_id, self::META_UPDATED_BY, get_current_user_id() );
        }

        // Optional: generate on publish is OFF by default and not implemented in v1.0.2.
    }

    public static function output_meta_description() : void {
        if ( is_admin() || ! is_singular() ) {
            return;
        }

        $post_id = get_queried_object_id();
        if ( ! $post_id ) {
            return;
        }

        $desc = get_post_meta( $post_id, self::META_DESC, true );
        $desc = is_string( $desc ) ? trim( $desc ) : '';
        if ( $desc === '' ) {
            return;
        }

        echo '<meta name="description" content="' . esc_attr( $desc ) . '">' . "\n";
    }

    public static function ajax_generate() : void {
        if ( ! self::can_use() ) {
            wp_send_json_error( array( 'message' => 'Not allowed.' ), 403 );
        }

        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, self::NONCE_AJAX ) ) {
            wp_send_json_error( array( 'message' => 'Bad nonce.' ), 400 );
        }

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) ) {
            wp_send_json_error( array( 'message' => 'Invalid post.' ), 400 );
        }

        $mode = isset( $_POST['mode'] ) ? sanitize_key( wp_unslash( $_POST['mode'] ) ) : 'generate';
        if ( ! in_array( $mode, array( 'generate', 'regenerate' ), true ) ) {
            $mode = 'generate';
        }

        $lock = get_post_meta( $post_id, self::META_LOCK, true );
        if ( $lock === '1' && $mode === 'generate' ) {
            wp_send_json_error( array( 'message' => 'Locked. Unlock to regenerate.' ), 409 );
        }

        $api_key = self::get_option_str( self::OPT_API_KEY, '' );
        if ( $api_key === '' ) {
            wp_send_json_error( array( 'message' => 'Missing API key.' ), 400 );
        }

        $result = self::generate_for_post( $post_id, $api_key );
        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ), 500 );
        }

        update_post_meta( $post_id, self::META_DESC, $result );
        update_post_meta( $post_id, self::META_UPDATED_AT, time() );
        update_post_meta( $post_id, self::META_UPDATED_BY, get_current_user_id() );

        wp_send_json_success( array(
            'description' => $result,
        ) );
    }

    public static function ajax_bulk_init() : void {
        if ( ! self::can_use() ) {
            wp_send_json_error( array( 'message' => 'Not allowed.' ), 403 );
        }

        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, self::NONCE_AJAX ) ) {
            wp_send_json_error( array( 'message' => 'Bad nonce.' ), 400 );
        }

        $types = isset( $_POST['post_types'] ) && is_array( $_POST['post_types'] ) ? (array) wp_unslash( $_POST['post_types'] ) : array();
        $types = array_values( array_filter( array_map( 'sanitize_key', $types ) ) );
        if ( ! $types ) {
            $types = self::get_allowed_post_types();
        }

        $include_existing = isset( $_POST['include_existing'] ) ? sanitize_text_field( wp_unslash( $_POST['include_existing'] ) ) : '';
        $include_existing = ( $include_existing === '1' );

        $args = array(
            'post_type'      => $types,
            'post_status'    => array( 'publish', 'draft', 'pending', 'future', 'private' ),
            'posts_per_page' => 5000,
            'fields'         => 'ids',
        );

        $ids = get_posts( $args );
        if ( ! is_array( $ids ) ) {
            $ids = array();
        }

        $queue = array();
        foreach ( $ids as $id ) {
            $id = intval( $id );
            if ( ! $id ) {
                continue;
            }
            if ( ! current_user_can( 'edit_post', $id ) ) {
                continue;
            }

            $lock = get_post_meta( $id, self::META_LOCK, true );
            if ( $lock === '1' ) {
                continue;
            }

            $existing = get_post_meta( $id, self::META_DESC, true );
            $has_existing = is_string( $existing ) && trim( $existing ) !== '';

            if ( $has_existing && ! $include_existing ) {
                continue;
            }

            $queue[] = $id;
        }

        $token = wp_generate_uuid4();
        $key   = self::BULK_TRANSIENT_PREFIX . $token;

        set_transient( $key, array(
            'queue'     => $queue,
            'total'     => count( $queue ),
            'processed' => 0,
            'started'   => time(),
        ), HOUR_IN_SECONDS );

        wp_send_json_success( array(
            'token' => $token,
            'total' => count( $queue ),
        ) );
    }

    public static function ajax_bulk_process() : void {
        if ( ! self::can_use() ) {
            wp_send_json_error( array( 'message' => 'Not allowed.' ), 403 );
        }

        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, self::NONCE_AJAX ) ) {
            wp_send_json_error( array( 'message' => 'Bad nonce.' ), 400 );
        }

        $token = isset( $_POST['token'] ) ? sanitize_text_field( wp_unslash( $_POST['token'] ) ) : '';
        if ( $token === '' ) {
            wp_send_json_error( array( 'message' => 'Missing token.' ), 400 );
        }

        $batch_size = isset( $_POST['batch_size'] ) ? intval( $_POST['batch_size'] ) : 5;
        if ( $batch_size < 1 ) $batch_size = 1;
        if ( $batch_size > 20 ) $batch_size = 20;

        $key  = self::BULK_TRANSIENT_PREFIX . $token;
        $data = get_transient( $key );
        if ( ! is_array( $data ) || ! isset( $data['queue'] ) || ! is_array( $data['queue'] ) ) {
            wp_send_json_error( array( 'message' => 'Bulk job not found or expired. Please start again.' ), 404 );
        }

        $api_key = self::get_option_str( self::OPT_API_KEY, '' );
        if ( $api_key === '' ) {
            wp_send_json_error( array( 'message' => 'Missing API key.' ), 400 );
        }

        $queue = $data['queue'];
        $processed = intval( $data['processed'] ?? 0 );
        $total = intval( $data['total'] ?? count( $queue ) );

        $done_now = 0;
        $errors = array();

        while ( $done_now < $batch_size && ! empty( $queue ) ) {
            $post_id = array_shift( $queue );
            $post_id = intval( $post_id );
            if ( ! $post_id ) {
                continue;
            }

            $result = self::generate_for_post( $post_id, $api_key );
            if ( is_wp_error( $result ) ) {
                $errors[] = 'Post ID ' . $post_id . ': ' . $result->get_error_message();
                // Stop the batch on rate limit or auth issues to avoid hammering.
                $code = $result->get_error_code();
                if ( in_array( $code, array( 'ofai_rate_limited', 'ofai_auth', 'ofai_bad_request' ), true ) ) {
                    // Put post back so user can resume later.
                    array_unshift( $queue, $post_id );
                    break;
                }
            } else {
                update_post_meta( $post_id, self::META_DESC, $result );
                update_post_meta( $post_id, self::META_UPDATED_AT, time() );
                update_post_meta( $post_id, self::META_UPDATED_BY, get_current_user_id() );
            }

            $done_now++;
            $processed++;
        }

        $data['queue'] = $queue;
        $data['processed'] = $processed;

        set_transient( $key, $data, HOUR_IN_SECONDS );

        $remaining = count( $queue );
        $finished = ( $remaining === 0 );

        if ( $finished ) {
            delete_transient( $key );
        }

        wp_send_json_success( array(
            'processed' => $processed,
            'total'     => $total,
            'remaining' => $remaining,
            'finished'  => $finished,
            'errors'    => $errors,
        ) );
    }

    public static function generate_for_post( int $post_id, string $api_key ) {
        $post = get_post( $post_id );
        if ( ! $post instanceof WP_Post ) {
            return new WP_Error( 'ofai_invalid_post', 'Post not found.' );
        }

        $title = get_the_title( $post_id );
        $title = is_string( $title ) ? $title : '';

        $content = get_post_field( 'post_content', $post_id );
        $content = is_string( $content ) ? $content : '';
        $content = strip_shortcodes( $content );
        $content = wp_strip_all_tags( $content );
        $content = html_entity_decode( $content, ENT_QUOTES, 'UTF-8' );
        $content = preg_replace( '/\s+/u', ' ', $content );
        $content = trim( $content );

        $max_words = self::get_option_int( self::OPT_MAX_WORDS, 800 );
        if ( $max_words < 50 ) $max_words = 50;
        if ( $max_words > 2000 ) $max_words = 2000;

        $excerpt_words = self::limit_words( $content, $max_words );

        $tone = self::get_option_str( self::OPT_TONE, 'neutral' );
        $tone_text = 'neutral';
        if ( $tone === 'professional' ) $tone_text = 'professional';
        if ( $tone === 'friendly' ) $tone_text = 'friendly';

        $limit = self::get_option_int( self::OPT_CHAR_LIMIT, 155 );
        if ( $limit < 80 ) $limit = 80;
        if ( $limit > 320 ) $limit = 320;

        $post_type_obj = get_post_type_object( $post->post_type );
        $post_type_label = $post_type_obj && ! empty( $post_type_obj->labels->singular_name )
            ? $post_type_obj->labels->singular_name
            : 'page';

        $prompt = self::build_prompt( $title, $post_type_label, $excerpt_words, $limit, $tone_text );

        $model = self::get_option_str( self::OPT_MODEL, 'gpt-4o-mini' );
        $model = $model ? $model : 'gpt-4o-mini';

        $ai_text = self::call_openai( $api_key, $model, $prompt );
        if ( is_wp_error( $ai_text ) ) {
            return $ai_text;
        }

        $final = self::format_meta( $ai_text, $limit );

        // Ensure we always have something sensible.
        if ( trim( $final ) === '' ) {
            $fallback = $title ? $title : get_bloginfo( 'name' );
            $final = self::format_meta( $fallback, $limit );
        }

        return $final;
    }

    public static function build_prompt( string $title, string $type_label, string $content_excerpt, int $limit, string $tone ) : string {
        $title = trim( $title );
        $content_excerpt = trim( $content_excerpt );

        $rules = array(
            'Write a meta description in English (UK).',
            'Make it sound natural and human.',
            'Use one sentence only.',
            'End with a full stop.',
            'Do not use em dashes.',
            'Do not use quotation marks.',
            'Keep it within ' . intval( $limit ) . ' characters.',
            'Make it specific to this ' . $type_label . '.',
        );

        // Add a small uniqueness cue (safe, non-personal).
        $rules[] = 'Avoid generic phrases and avoid repeating the title exactly.';

        $prompt = "Title: {$title}\n";
        $prompt .= "Content excerpt:\n{$content_excerpt}\n\n";
        $prompt .= "Tone: {$tone}\n\n";
        $prompt .= "Rules:\n- " . implode( "\n- ", $rules ) . "\n\n";
        $prompt .= "Return only the meta description text.";

        return $prompt;
    }

    public static function call_openai( string $api_key, string $model, string $prompt ) {
        $api_key = trim( $api_key );
        if ( $api_key === '' ) {
            return new WP_Error( 'ofai_auth', 'OpenAI API key is missing.' );
        }

        $headers = array(
            'Content-Type'  => 'application/json; charset=utf-8',
            'Authorization' => 'Bearer ' . $api_key,
        );

        // Try Responses API first.
        $responses_url = 'https://api.openai.com/v1/responses';
        $body = array(
            'model' => $model,
            'input' => array(
                array(
                    'role'    => 'user',
                    'content' => array(
                        array(
                            'type' => 'text',
                            'text' => $prompt,
                        ),
                    ),
                ),
            ),
        );

        $res = wp_remote_post( $responses_url, array(
            'timeout' => 25,
            'headers' => $headers,
            'body'    => wp_json_encode( $body, JSON_UNESCAPED_UNICODE ),
        ) );

        $text = self::parse_openai_response_text( $res, 'responses' );
        if ( is_wp_error( $text ) ) {
            // If endpoint is missing or parsing fails, try Chat Completions as fallback.
            $chat_url = 'https://api.openai.com/v1/chat/completions';
            $body2 = array(
                'model' => $model,
                'messages' => array(
                    array(
                        'role'    => 'user',
                        'content' => $prompt,
                    ),
                ),
                'temperature' => 0.6,
            );

            $res2 = wp_remote_post( $chat_url, array(
                'timeout' => 25,
                'headers' => $headers,
                'body'    => wp_json_encode( $body2, JSON_UNESCAPED_UNICODE ),
            ) );

            $text2 = self::parse_openai_response_text( $res2, 'chat' );
            if ( is_wp_error( $text2 ) ) {
                // Return original error if it is more specific.
                return $text2;
            }
            return $text2;
        }

        return $text;
    }

    public static function parse_openai_response_text( $res, string $mode ) {
        if ( is_wp_error( $res ) ) {
            return new WP_Error( 'ofai_http', $res->get_error_message() );
        }

        $code = wp_remote_retrieve_response_code( $res );
        $body = wp_remote_retrieve_body( $res );

        if ( $code === 401 || $code === 403 ) {
            return new WP_Error( 'ofai_auth', 'OpenAI authorisation failed. Please check your API key.' );
        }
        if ( $code === 429 ) {
            return new WP_Error( 'ofai_rate_limited', 'Rate limited by OpenAI. Please try again shortly.' );
        }
        if ( $code < 200 || $code >= 300 ) {
            $msg = 'OpenAI request failed.';
            if ( $body ) {
                $decoded = json_decode( $body, true );
                if ( is_array( $decoded ) && ! empty( $decoded['error']['message'] ) ) {
                    $msg = sanitize_text_field( (string) $decoded['error']['message'] );
                }
            }
            return new WP_Error( 'ofai_bad_request', $msg );
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE || ! is_array( $data ) ) {
            return new WP_Error( 'ofai_parse', 'Could not parse OpenAI response.' );
        }

        $text = '';

        if ( $mode === 'responses' ) {
            // Prefer output_text if present.
            if ( isset( $data['output_text'] ) && is_string( $data['output_text'] ) ) {
                $text = $data['output_text'];
            } elseif ( isset( $data['output'] ) && is_array( $data['output'] ) ) {
                foreach ( $data['output'] as $o ) {
                    if ( isset( $o['content'] ) && is_array( $o['content'] ) ) {
                        foreach ( $o['content'] as $c ) {
                            if ( isset( $c['type'] ) && $c['type'] === 'output_text' && isset( $c['text'] ) ) {
                                $text .= (string) $c['text'];
                            } elseif ( isset( $c['type'] ) && $c['type'] === 'text' && isset( $c['text'] ) ) {
                                $text .= (string) $c['text'];
                            }
                        }
                    }
                }
            }
        } else {
            // chat completions
            $text = (string) ( $data['choices'][0]['message']['content'] ?? '' );
        }

        $text = is_string( $text ) ? trim( $text ) : '';
        if ( $text === '' ) {
            return new WP_Error( 'ofai_empty', 'OpenAI returned an empty result.' );
        }

        return $text;
    }

    public static function limit_words( string $text, int $max_words ) : string {
        $text = trim( $text );
        if ( $text === '' ) {
            return '';
        }
        $parts = preg_split( '/\s+/u', $text );
        if ( ! is_array( $parts ) ) {
            return $text;
        }
        if ( count( $parts ) <= $max_words ) {
            return $text;
        }
        $parts = array_slice( $parts, 0, $max_words );
        return trim( implode( ' ', $parts ) );
    }

    public static function format_meta( string $text, int $limit = 155 ) : string {
        $text = wp_strip_all_tags( (string) $text );
        $text = html_entity_decode( $text, ENT_QUOTES, 'UTF-8' );
        $text = preg_replace( '/\s+/u', ' ', $text );
        $text = trim( $text );

        // Remove wrapping quotes if present.
        $text = preg_replace( '/^["\']+|["\']+$/u', '', $text );
        $text = trim( $text );

        // Remove em dashes and similar long dashes.
        $text = str_replace( array( '—', '–' ), '-', $text );

        if ( mb_strlen( $text, 'UTF-8' ) > $limit ) {
            $cut = mb_substr( $text, 0, $limit, 'UTF-8' );

            // Prefer to end on sentence punctuation within the cut.
            $last_punct = max(
                self::mb_strrpos_any( $cut, array( '.', '!', '?' ) ),
                self::mb_strrpos_any( $cut, array( ';', ':' ) )
            );

            if ( $last_punct !== -1 && $last_punct > 40 ) {
                $cut = mb_substr( $cut, 0, $last_punct + 1, 'UTF-8' );
            } else {
                // Cut to last space and add a full stop.
                $pos = mb_strrpos( $cut, ' ', 0, 'UTF-8' );
                if ( $pos !== false ) {
                    $cut = mb_substr( $cut, 0, $pos, 'UTF-8' );
                }
                $cut = rtrim( $cut, " ,;:—–-" ) . '.';
            }
            $text = $cut;
        }

        $text = trim( $text );
        if ( $text === '' ) {
            return '';
        }

        if ( ! preg_match( '/[.!?]$/u', $text ) ) {
            $text .= '.';
        }

        // Final clean.
        $text = preg_replace( '/\s+/u', ' ', $text );
        $text = trim( $text );

        // Avoid double full stops.
        $text = preg_replace( '/\.\.+$/u', '.', $text );

        return $text;
    }

    private static function mb_strrpos_any( string $haystack, array $needles ) : int {
        $best = -1;
        foreach ( $needles as $n ) {
            $p = mb_strrpos( $haystack, $n, 0, 'UTF-8' );
            if ( $p !== false && $p > $best ) {
                $best = $p;
            }
        }
        return $best;
    }

    public static function settings_page() : void {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $api_key = self::get_option_str( self::OPT_API_KEY, '' );
        $model   = self::get_option_str( self::OPT_MODEL, 'gpt-4o-mini' );
        $limit   = self::get_option_int( self::OPT_CHAR_LIMIT, 155 );
        $words   = self::get_option_int( self::OPT_MAX_WORDS, 800 );
        $tone    = self::get_option_str( self::OPT_TONE, 'neutral' );
        $gen_pub = self::get_option_str( self::OPT_GEN_ON_PUBLISH, 'off' );
        $allowed = self::get_allowed_post_types();
        $all_types = get_post_types( array( 'public' => true ), 'objects' );
        ?>
        <div class="wrap">
            <h1>OtterFixer Simple AI Meta Descriptions Generator</h1>

            <p>
                This plugin generates meta descriptions using the OpenAI API. The API key is provided by the site owner.
                Generated descriptions are saved per post, and will not be regenerated unless you click generate again.
            </p>

            <form method="post" action="options.php">
                <?php settings_fields( 'ofai_settings' ); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="ofai_openai_api_key">OpenAI API key</label></th>
                        <td>
                            <input type="password" id="ofai_openai_api_key" name="<?php echo esc_attr( self::OPT_API_KEY ); ?>" value="<?php echo esc_attr( $api_key ); ?>" class="regular-text" autocomplete="off" />
                            <p class="description">Your API key is stored in your WordPress database. Do not share it.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="ofai_openai_model">Model</label></th>
                        <td>
                            <input type="text" id="ofai_openai_model" name="<?php echo esc_attr( self::OPT_MODEL ); ?>" value="<?php echo esc_attr( $model ); ?>" class="regular-text" />
                            <p class="description">Example: gpt-4o-mini. The model must be available to your API key.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="ofai_char_limit">Character limit</label></th>
                        <td>
                            <input type="number" id="ofai_char_limit" name="<?php echo esc_attr( self::OPT_CHAR_LIMIT ); ?>" value="<?php echo esc_attr( $limit ); ?>" min="80" max="320" />
                            <p class="description">Typical meta descriptions are around 150 to 160 characters.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="ofai_max_words">Max words to send</label></th>
                        <td>
                            <input type="number" id="ofai_max_words" name="<?php echo esc_attr( self::OPT_MAX_WORDS ); ?>" value="<?php echo esc_attr( $words ); ?>" min="50" max="2000" />
                            <p class="description">This is the maximum number of words taken from the page content for generation. 800 is usually plenty.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Tone</th>
                        <td>
                            <select name="<?php echo esc_attr( self::OPT_TONE ); ?>">
                                <option value="neutral" <?php selected( $tone, 'neutral' ); ?>>Neutral</option>
                                <option value="professional" <?php selected( $tone, 'professional' ); ?>>Professional</option>
                                <option value="friendly" <?php selected( $tone, 'friendly' ); ?>>Friendly</option>
                            </select>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Generate on publish</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr( self::OPT_GEN_ON_PUBLISH ); ?>" value="on" <?php checked( $gen_pub, 'on' ); ?> />
                                Enable generate on publish (off by default)
                            </label>
                            <p class="description">Recommended off. Bulk generate and manual generate are safer and avoid surprise API usage.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Post types</th>
                        <td>
                            <?php foreach ( $all_types as $pt => $obj ) : ?>
                                <label style="display:block;margin:0 0 6px;">
                                    <input type="checkbox" name="<?php echo esc_attr( self::OPT_ALLOWED_TYPES ); ?>[]" value="<?php echo esc_attr( $pt ); ?>" <?php checked( in_array( $pt, $allowed, true ) ); ?> />
                                    <?php echo esc_html( $obj->labels->singular_name ); ?>
                                </label>
                            <?php endforeach; ?>
                            <p class="description">Select which post types should show the meta box and be included in bulk generation.</p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( 'Save settings' ); ?>
            </form>

            <hr />

            <h2>Bulk generator</h2>
            <p>
                Use the bulk generator here:
                <a href="<?php echo esc_url( admin_url( 'tools.php?page=ofai-meta-bulk' ) ); ?>">Tools, AI Meta Descriptions</a>
            </p>

            <h2>Privacy note</h2>
            <p>
                When you generate a meta description, the plugin sends the post title and a cleaned excerpt of the post content to OpenAI.
                Do not use this plugin for content that should not be shared with third parties.
            </p>
        </div>
        <?php
    }

    public static function bulk_page() : void {
        if ( ! self::can_use() ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'otterfixer-simple-ai-meta-descriptions-generator' ) );
        }

        $api_key = self::get_option_str( self::OPT_API_KEY, '' );
        $allowed = self::get_allowed_post_types();
        $all_types = get_post_types( array( 'public' => true ), 'objects' );
        ?>
        <div class="wrap">
            <h1>AI Meta Descriptions (Bulk)</h1>

            <?php if ( $api_key === '' ) : ?>
                <div class="notice notice-warning">
                    <p>
                        OpenAI API key is missing. Please add it in <a href="<?php echo esc_url( admin_url( 'options-general.php?page=ofai-meta-settings' ) ); ?>">Settings</a>.
                    </p>
                </div>
            <?php endif; ?>

            <p>
                Bulk generation runs in small batches to avoid timeouts and rate limits. It saves each description to the post so it only generates once.
            </p>

            <div class="ofai-bulk-box">
                <h2>Bulk generate</h2>

                <p><strong>Post types</strong></p>
                <?php foreach ( $all_types as $pt => $obj ) : ?>
                    <label style="display:block;margin:0 0 6px;">
                        <input type="checkbox" class="ofai-bulk-type" value="<?php echo esc_attr( $pt ); ?>" <?php checked( in_array( $pt, $allowed, true ) ); ?> />
                        <?php echo esc_html( $obj->labels->singular_name ); ?>
                    </label>
                <?php endforeach; ?>

                <p style="margin-top:12px;">
                    <label>
                        <input type="checkbox" id="ofai-include-existing" value="1" />
                        Regenerate existing descriptions
                    </label>
                </p>

                <p style="margin-top:12px;">
                    <label>
                        Batch size
                        <select id="ofai-batch-size">
                            <option value="3">3</option>
                            <option value="5" selected>5</option>
                            <option value="8">8</option>
                            <option value="10">10</option>
                        </select>
                    </label>
                </p>

                <p style="margin-top:14px;">
                    <button type="button" class="button button-primary" id="ofai-bulk-start">Generate meta descriptions</button>
                    <button type="button" class="button" id="ofai-bulk-stop" disabled>Stop</button>
                </p>

                <div class="ofai-progress">
                    <div class="ofai-progress-bar" style="width:0%"></div>
                </div>
                <p class="ofai-progress-text" aria-live="polite"></p>

                <div class="ofai-bulk-errors" style="display:none;">
                    <h3>Errors</h3>
                    <ul class="ofai-bulk-errors-list"></ul>
                </div>
            </div>
        </div>
        <?php
    }
}

OFAI_Meta_Descriptions::init();
