<?php
/*
Plugin Name: OtterFixer Orgainsed Plugin Notes
Description: Add private notes to each installed plugin so you always know why it is installed. Includes a "Missing notes" filter, a bulk notes page, and visibility settings.
Version: 1.1.0
Author: OtterFixer
License: GPLv2 or later
Text Domain: otterfixer-organised-plugin-notes
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

final class OFOPN_Plugin_Notes {

    const OPT_NOTES      = 'ofopn_plugin_notes';
    const OPT_VISIBILITY = 'ofopn_visibility'; // admin | editor

    const NONCE_AJAX_ACTION = 'ofopn_save_note';
    const NONCE_VIEW_ACTION = 'ofopn_plugins_view';
    const NONCE_BULK_ACTION = 'ofopn_bulk_save';

    public static function init() : void {
        add_action( 'admin_menu', array( __CLASS__, 'admin_menu' ) );
        add_action( 'admin_init', array( __CLASS__, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_assets' ) );

        // Plugins list UI.
        add_action( 'after_plugin_row', array( __CLASS__, 'render_note_row' ), 10, 3 );
        add_filter( 'views_plugins', array( __CLASS__, 'add_missing_notes_view' ) );
        add_filter( 'all_plugins', array( __CLASS__, 'filter_plugins_missing_notes' ) );

        // AJAX save.
        add_action( 'wp_ajax_ofopn_save_note', array( __CLASS__, 'ajax_save_note' ) );
    }

    public static function get_required_cap() : string {
        $visibility = get_option( self::OPT_VISIBILITY, 'admin' );
        // "editor" means editors can view and edit notes too.
        return ( $visibility === 'editor' ) ? 'edit_pages' : 'manage_options';
    }

    public static function can_user_access() : bool {
        return current_user_can( self::get_required_cap() );
    }

    public static function admin_menu() : void {
        add_options_page(
            'Plugin Notes',
            'Plugin Notes',
            'manage_options',
            'ofopn-plugin-notes',
            array( __CLASS__, 'settings_page' )
        );

        // Bulk page under Plugins menu, because it's directly related.
        add_plugins_page(
            'Plugin Notes (Bulk)',
            'Plugin Notes (Bulk)',
            self::get_required_cap(),
            'ofopn-bulk',
            array( __CLASS__, 'bulk_page' )
        );
    }

    public static function register_settings() : void {
        register_setting(
            'ofopn_settings',
            self::OPT_VISIBILITY,
            array(
                'type'              => 'string',
                'sanitize_callback' => array( __CLASS__, 'sanitize_visibility' ),
                'default'           => 'admin',
            )
        );
    }

    public static function sanitize_visibility( $value ) : string {
        $value = is_string( $value ) ? strtolower( $value ) : 'admin';
        return in_array( $value, array( 'admin', 'editor' ), true ) ? $value : 'admin';
    }

    public static function enqueue_assets( string $hook ) : void {
        if ( ! self::can_user_access() ) {
            return;
        }

        if ( $hook === 'plugins.php' ) {
            wp_enqueue_style(
                'ofopn-admin',
                plugins_url( 'assets/ofopn-admin.css', __FILE__ ),
                array(),
                '1.1.0'
            );

            wp_enqueue_script(
                'ofopn-admin',
                plugins_url( 'assets/ofopn-admin.js', __FILE__ ),
                array( 'jquery' ),
                '1.1.0',
                true
            );

            wp_localize_script(
                'ofopn-admin',
                'OFOPN',
                array(
                    'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                    'nonce'   => wp_create_nonce( self::NONCE_AJAX_ACTION ),
                    'i18n'    => array(
                        'saving'   => 'Saving...',
                        'saved'    => 'Saved',
                        'saveFail' => 'Could not save. Please refresh and try again.',
                    ),
                )
            );
        }
    }

    /**
     * Notes are stored as:
     * [ 'plugin/file.php' => [ 'note' => string, 'required_by_theme' => 0/1, 'pricing' => free|paid|unknown, 'updated_at' => int, 'updated_by' => int ] ]
     */
    public static function get_notes() : array {
        $notes = get_option( self::OPT_NOTES, array() );
        return is_array( $notes ) ? $notes : array();
    }

    public static function normalize_note_row( array $row ) : array {
        return array(
            'note'              => isset( $row['note'] ) ? (string) $row['note'] : '',
            'required_by_theme' => ! empty( $row['required_by_theme'] ) ? 1 : 0,
            'pricing'           => isset( $row['pricing'] ) ? (string) $row['pricing'] : 'unknown',
            'updated_at'        => isset( $row['updated_at'] ) ? intval( $row['updated_at'] ) : 0,
            'updated_by'        => isset( $row['updated_by'] ) ? intval( $row['updated_by'] ) : 0,
        );
    }

    public static function get_note_for( string $plugin_file ) : array {
        $notes = self::get_notes();
        if ( isset( $notes[ $plugin_file ] ) && is_array( $notes[ $plugin_file ] ) ) {
            return self::normalize_note_row( $notes[ $plugin_file ] );
        }

        return self::normalize_note_row( array() );
    }

    public static function sanitize_pricing( string $pricing ) : string {
        $pricing = strtolower( trim( $pricing ) );
        return in_array( $pricing, array( 'free', 'paid', 'unknown' ), true ) ? $pricing : 'unknown';
    }

    public static function sanitize_bool( $value ) : int {
        return ! empty( $value ) ? 1 : 0;
    }

    public static function set_note_for( string $plugin_file, string $note, int $required_by_theme, string $pricing ) : bool {
        $notes = self::get_notes();

        $notes[ $plugin_file ] = array(
            'note'              => $note,
            'required_by_theme' => $required_by_theme ? 1 : 0,
            'pricing'           => $pricing,
            'updated_at'        => time(),
            'updated_by'        => get_current_user_id(),
        );

        return update_option( self::OPT_NOTES, $notes, false );
    }

    public static function format_updated_meta( array $note_data ) : array {
        $updated_at = isset( $note_data['updated_at'] ) ? intval( $note_data['updated_at'] ) : 0;
        $updated_by = isset( $note_data['updated_by'] ) ? intval( $note_data['updated_by'] ) : 0;

        $updated_by_name = '';
        if ( $updated_by > 0 ) {
            $user = get_user_by( 'id', $updated_by );
            if ( $user && ! is_wp_error( $user ) ) {
                $updated_by_name = $user->display_name;
            }
        }

        $updated_at_str = '';
        if ( $updated_at > 0 ) {
            $updated_at_str = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $updated_at );
        }

        return array( $updated_at_str, $updated_by_name );
    }

    public static function render_note_row( string $plugin_file, array $plugin_data, string $status ) : void {
        if ( ! self::can_user_access() ) {
            return;
        }

        $note_data = self::get_note_for( $plugin_file );

        $note              = (string) $note_data['note'];
        $required_by_theme = ! empty( $note_data['required_by_theme'] );
        $pricing           = (string) $note_data['pricing'];

        list( $updated_at_str, $updated_by_name ) = self::format_updated_meta( $note_data );

        $colspan = 3; // default
        $screen  = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        if ( $screen && ! empty( $screen->id ) && $screen->id === 'plugins-network' ) {
            $colspan = 4;
        }

        $id_hash = md5( $plugin_file );
        ?>
        <tr class="ofopn-note-row" data-ofopn-plugin="<?php echo esc_attr( $plugin_file ); ?>">
            <td colspan="<?php echo intval( $colspan ); ?>" class="ofopn-note-cell">
                <div class="ofopn-wrap">
                    <div class="ofopn-left">
                        <label class="ofopn-label" for="<?php echo esc_attr( 'ofopn-note-' . $id_hash ); ?>">
                            Why is this installed?
                        </label>

                        <textarea
                            id="<?php echo esc_attr( 'ofopn-note-' . $id_hash ); ?>"
                            class="ofopn-textarea"
                            rows="2"
                            placeholder="Add a short note, for example: Required for Stripe payments, or Installed for site forms."
                        ><?php echo esc_textarea( $note ); ?></textarea>

                        <div class="ofopn-flags">
                            <label class="ofopn-flag">
                                <input type="checkbox" class="ofopn-required-by-theme" <?php checked( $required_by_theme, true ); ?> />
                                Required by theme
                            </label>

                            <label class="ofopn-flag">
                                Pricing:
                                <select class="ofopn-pricing">
                                    <option value="unknown" <?php selected( $pricing, 'unknown' ); ?>>Unknown</option>
                                    <option value="free" <?php selected( $pricing, 'free' ); ?>>Free</option>
                                    <option value="paid" <?php selected( $pricing, 'paid' ); ?>>Paid</option>
                                </select>
                            </label>

                            <a class="ofopn-bulk-link" href="<?php echo esc_url( admin_url( 'plugins.php?page=ofopn-bulk' ) ); ?>">Open bulk page</a>
                        </div>

                        <div class="ofopn-meta">
                            <?php if ( $updated_at_str ) : ?>
                                <span class="ofopn-meta-item">
                                    Last updated: <?php echo esc_html( $updated_at_str ); ?>
                                </span>
                            <?php endif; ?>
                            <?php if ( $updated_by_name ) : ?>
                                <span class="ofopn-meta-item">
                                    Updated by: <?php echo esc_html( $updated_by_name ); ?>
                                </span>
                            <?php endif; ?>
                            <span class="ofopn-status" aria-live="polite"></span>
                        </div>
                    </div>

                    <div class="ofopn-right">
                        <button type="button" class="button button-secondary ofopn-save">
                            Save
                        </button>
                    </div>
                </div>
            </td>
        </tr>
        <?php
    }

    public static function ajax_save_note() : void {
        if ( ! self::can_user_access() ) {
            wp_send_json_error( array( 'message' => 'Not allowed.' ), 403 );
        }

        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, self::NONCE_AJAX_ACTION ) ) {
            wp_send_json_error( array( 'message' => 'Bad nonce.' ), 400 );
        }

        $plugin_file = isset( $_POST['plugin_file'] ) ? sanitize_text_field( wp_unslash( $_POST['plugin_file'] ) ) : '';
        if ( $plugin_file === '' ) {
            wp_send_json_error( array( 'message' => 'Missing plugin file.' ), 400 );
        }

        $note_raw = isset( $_POST['note'] ) ? wp_unslash( $_POST['note'] ) : '';
        $note     = sanitize_textarea_field( $note_raw );

        $required_by_theme_raw = isset( $_POST['required_by_theme'] ) ? sanitize_text_field( wp_unslash( $_POST['required_by_theme'] ) ) : '';
        $required_by_theme     = self::sanitize_bool( $required_by_theme_raw );

        $pricing_raw = isset( $_POST['pricing'] ) ? sanitize_text_field( wp_unslash( $_POST['pricing'] ) ) : 'unknown';
        $pricing     = self::sanitize_pricing( $pricing_raw );

        // Limit length to prevent accidental huge saves.
        if ( strlen( $note ) > 2000 ) {
            $note = substr( $note, 0, 2000 );
        }

        $ok = self::set_note_for( $plugin_file, $note, $required_by_theme, $pricing );
        if ( ! $ok ) {
            wp_send_json_error( array( 'message' => 'Save failed.' ), 500 );
        }

        $note_data = self::get_note_for( $plugin_file );
        list( $updated_at_str, $updated_by_name ) = self::format_updated_meta( $note_data );

        wp_send_json_success(
            array(
                'note'              => $note,
                'required_by_theme' => $required_by_theme,
                'pricing'           => $pricing,
                'updated_at'        => $updated_at_str,
                'updated_by'        => $updated_by_name,
            )
        );
    }

    public static function get_view_nonce() : string {
        return wp_create_nonce( self::NONCE_VIEW_ACTION );
    }

    public static function is_missing_view_active() : bool {
        $flag = isset( $_GET['ofopn'] ) ? sanitize_key( wp_unslash( $_GET['ofopn'] ) ) : '';
        if ( $flag !== 'missing' ) {
            return false;
        }

        $nonce = isset( $_GET['ofopn_nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['ofopn_nonce'] ) ) : '';
        return ( $nonce && wp_verify_nonce( $nonce, self::NONCE_VIEW_ACTION ) );
    }

    public static function add_missing_notes_view( array $views ) : array {
        if ( ! self::can_user_access() ) {
            return $views;
        }

        $notes   = self::get_notes();
        $count   = 0;
        $plugins = get_plugins();

        foreach ( $plugins as $plugin_file => $data ) {
            $row  = isset( $notes[ $plugin_file ] ) && is_array( $notes[ $plugin_file ] ) ? self::normalize_note_row( $notes[ $plugin_file ] ) : self::normalize_note_row( array() );
            $note = trim( (string) $row['note'] );

            if ( $note === '' ) {
                $count++;
            }
        }

        $current = self::is_missing_view_active() ? 'current' : '';
        $url     = add_query_arg(
            array(
                'ofopn'       => 'missing',
                'ofopn_nonce' => self::get_view_nonce(),
            ),
            admin_url( 'plugins.php' )
        );

        $views['ofopn_missing'] = sprintf(
            '<a href="%s" class="%s">Missing notes <span class="count">(%d)</span></a>',
            esc_url( $url ),
            esc_attr( $current ),
            intval( $count )
        );

        return $views;
    }

    public static function filter_plugins_missing_notes( array $all_plugins ) : array {
        if ( ! self::can_user_access() ) {
            return $all_plugins;
        }

        if ( ! self::is_missing_view_active() ) {
            return $all_plugins;
        }

        $notes = self::get_notes();
        foreach ( $all_plugins as $plugin_file => $data ) {
            $row  = isset( $notes[ $plugin_file ] ) && is_array( $notes[ $plugin_file ] ) ? self::normalize_note_row( $notes[ $plugin_file ] ) : self::normalize_note_row( array() );
            $note = trim( (string) $row['note'] );

            if ( $note !== '' ) {
                unset( $all_plugins[ $plugin_file ] );
            }
        }

        return $all_plugins;
    }

    public static function settings_page() : void {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $visibility = get_option( self::OPT_VISIBILITY, 'admin' );
        ?>
        <div class="wrap">
            <h1>OtterFixer Orgainsed Plugin Notes</h1>

            <p>
                Add a private note to each installed plugin so your team always knows why it is installed.
                Notes appear on the Plugins page, under each plugin.
            </p>

            <form method="post" action="options.php">
                <?php settings_fields( 'ofopn_settings' ); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Who can view and edit notes?</th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="radio" name="<?php echo esc_attr( self::OPT_VISIBILITY ); ?>" value="admin" <?php checked( $visibility, 'admin' ); ?> />
                                    Admins only
                                </label>
                                <br />
                                <label>
                                    <input type="radio" name="<?php echo esc_attr( self::OPT_VISIBILITY ); ?>" value="editor" <?php checked( $visibility, 'editor' ); ?> />
                                    Admins and Editors
                                </label>
                                <p class="description">
                                    If you choose "Admins and Editors", anyone with the "edit_pages" capability can view and edit notes.
                                </p>
                            </fieldset>
                        </td>
                    </tr>
                </table>

                <?php submit_button( 'Save settings' ); ?>
            </form>

            <hr />

            <h2>Bulk page</h2>
            <p>
                You can manage all plugin notes on one page here:
                <a href="<?php echo esc_url( admin_url( 'plugins.php?page=ofopn-bulk' ) ); ?>">Plugin Notes (Bulk)</a>
            </p>
        </div>
        <?php
    }

    public static function bulk_page() : void {
        if ( ! self::can_user_access() ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'otterfixer-organised-plugin-notes' ) );
        }

        $did_save = false;
        $error    = '';

        if ( isset( $_POST['ofopn_bulk_nonce'] ) ) {
            $nonce = sanitize_text_field( wp_unslash( $_POST['ofopn_bulk_nonce'] ) );
            if ( ! wp_verify_nonce( $nonce, self::NONCE_BULK_ACTION ) ) {
                $error = 'Security check failed. Please try again.';
            } else {
                $posted = isset( $_POST['ofopn'] ) ? (array) $_POST['ofopn'] : array();
                $posted = wp_unslash( $posted );

                $plugins = get_plugins();

                foreach ( $plugins as $plugin_file => $data ) {
                    $row = isset( $posted[ $plugin_file ] ) && is_array( $posted[ $plugin_file ] ) ? $posted[ $plugin_file ] : array();

                    $note_raw = isset( $row['note'] ) ? $row['note'] : '';
                    $note     = sanitize_textarea_field( $note_raw );

                    $required_by_theme = isset( $row['required_by_theme'] ) ? 1 : 0;

                    $pricing_raw = isset( $row['pricing'] ) ? (string) $row['pricing'] : 'unknown';
                    $pricing     = self::sanitize_pricing( sanitize_text_field( $pricing_raw ) );

                    if ( strlen( $note ) > 2000 ) {
                        $note = substr( $note, 0, 2000 );
                    }

                    // Only save if something is present or already exists. This avoids huge options for totally empty installs.
                    $existing = self::get_note_for( $plugin_file );
                    $has_any  = ( trim( $note ) !== '' ) || $required_by_theme || ( $pricing !== 'unknown' ) || ( trim( (string) $existing['note'] ) !== '' ) || ! empty( $existing['required_by_theme'] ) || ( (string) $existing['pricing'] !== 'unknown' );

                    if ( $has_any ) {
                        self::set_note_for( $plugin_file, $note, $required_by_theme, $pricing );
                    }
                }

                $did_save = true;
            }
        }

        $notes   = self::get_notes();
        $plugins = get_plugins();
        ?>
        <div class="wrap">
            <h1>Plugin Notes (Bulk)</h1>

            <p>
                Manage all notes in one place. Changes here update the notes shown on the Plugins page.
            </p>

            <?php if ( $did_save ) : ?>
                <div class="notice notice-success is-dismissible"><p>Saved.</p></div>
            <?php endif; ?>

            <?php if ( $error ) : ?>
                <div class="notice notice-error"><p><?php echo esc_html( $error ); ?></p></div>
            <?php endif; ?>

            <form method="post">
                <input type="hidden" name="ofopn_bulk_nonce" value="<?php echo esc_attr( wp_create_nonce( self::NONCE_BULK_ACTION ) ); ?>" />

                <table class="widefat striped ofopn-bulk-table">
                    <thead>
                        <tr>
                            <th style="width: 22%;">Plugin</th>
                            <th>Note</th>
                            <th style="width: 14%;">Required by theme</th>
                            <th style="width: 12%;">Pricing</th>
                            <th style="width: 18%;">Last updated</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $plugins as $plugin_file => $data ) : ?>
                            <?php
                                $row = isset( $notes[ $plugin_file ] ) && is_array( $notes[ $plugin_file ] ) ? self::normalize_note_row( $notes[ $plugin_file ] ) : self::normalize_note_row( array() );
                                list( $updated_at_str, $updated_by_name ) = self::format_updated_meta( $row );
                                $meta = trim( $updated_at_str . ( $updated_by_name ? ' by ' . $updated_by_name : '' ) );
                            ?>
                            <tr>
                                <td>
                                    <strong><?php echo esc_html( $data['Name'] ); ?></strong><br />
                                    <code><?php echo esc_html( $plugin_file ); ?></code>
                                </td>
                                <td>
                                    <textarea
                                        class="large-text"
                                        rows="2"
                                        name="<?php echo esc_attr( 'ofopn[' . $plugin_file . '][note]' ); ?>"
                                        placeholder="Why is this installed?"
                                    ><?php echo esc_textarea( (string) $row['note'] ); ?></textarea>
                                </td>
                                <td style="text-align: center;">
                                    <label>
                                        <input type="checkbox" name="<?php echo esc_attr( 'ofopn[' . $plugin_file . '][required_by_theme]' ); ?>" value="1" <?php checked( ! empty( $row['required_by_theme'] ) ); ?> />
                                    </label>
                                </td>
                                <td>
                                    <select name="<?php echo esc_attr( 'ofopn[' . $plugin_file . '][pricing]' ); ?>">
                                        <option value="unknown" <?php selected( (string) $row['pricing'], 'unknown' ); ?>>Unknown</option>
                                        <option value="free" <?php selected( (string) $row['pricing'], 'free' ); ?>>Free</option>
                                        <option value="paid" <?php selected( (string) $row['pricing'], 'paid' ); ?>>Paid</option>
                                    </select>
                                </td>
                                <td>
                                    <?php echo $meta ? esc_html( $meta ) : '—'; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <?php submit_button( 'Save all notes' ); ?>
            </form>
        </div>
        <?php
    }
}

OFOPN_Plugin_Notes::init();
