<?php
/**
 * Plugin Name: OtterFixer AI Bot Tracker
 * Plugin URI:  https://otterfixer.com/free-wordpress-plugins/otterfixer-ai-bot-tracker/
 * Description: Logs visits from common AI/LLM crawler user agents (such as GPTBot and ClaudeBot) and shows a simple, local-only report in wp-admin.
 * Version:     1.0.5
 * Author:      OtterFixer
 * Author URI:  https://otterfixer.com/
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: otterfixer-ai-bot-tracker
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

final class OTTERFIXER_AIBT {
    const VERSION = '1.0.5';
    const SLUG    = 'otterfixer-ai-bot-tracker';
    const TABLE   = 'otterfixer_aibt_aibt_hits';

    /**
     * Capability required to VIEW the report.
     * Keep this consistent everywhere so we don't end up with "not allowed" surprises.
     */
    const CAP_VIEW  = 'manage_options';

    /**
     * Capability required to CHANGE settings / clear logs.
     * (Same as view for v1 to keep things simple and predictable.)
     */
    const CAP_ADMIN = 'manage_options';

    public static function init() : void {
        add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( __CLASS__, 'plugin_action_links' ) );
        add_action( 'admin_notices', array( __CLASS__, 'admin_help_notice' ) );
        register_activation_hook( __FILE__, array( __CLASS__, 'on_activate' ) );
        register_uninstall_hook( __FILE__, array( __CLASS__, 'on_uninstall' ) );

        add_action( 'init', array( __CLASS__, 'maybe_log_hit' ), 1 );
        add_action( 'admin_menu', array( __CLASS__, 'admin_menu' ) );

        // Optional debug notice: add ?otterfixer_aibt_aibt_debug=1 to any wp-admin page to confirm the plugin file is loading.
    }

    public static function on_activate() : void {
        global $wpdb;

        $table = esc_sql( self::table_name() );
        $charset_collate = $wpdb->get_charset_collate();

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $sql = "CREATE TABLE {$table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            ts DATETIME NOT NULL,
            bot VARCHAR(80) NOT NULL DEFAULT '',
            path TEXT NULL,
            referer TEXT NULL,
            ip VARCHAR(64) NOT NULL DEFAULT '',
            ua TEXT NULL,
            PRIMARY KEY  (id),
            KEY bot (bot),
            KEY ts (ts)
        ) {$charset_collate};";

        dbDelta( $sql );

        // Default bot patterns (editable later if needed).
        if ( false === get_option( 'otterfixer_aibt_bots', false ) ) {
            $default = array(
                'GPTBot'          => 'GPTBot',
                'ChatGPT-User'    => 'ChatGPT-User',
                'ClaudeBot'       => 'ClaudeBot',
                'PerplexityBot'   => 'PerplexityBot',
                'Google-Extended' => 'Google-Extended',
                'CCBot'           => 'CCBot', // Common Crawl
                'Applebot'        => 'Applebot',
                'Bytespider'      => 'Bytespider',
            );
            add_option( 'otterfixer_aibt_bots', $default, '', false );
        }
    }

    public static function on_uninstall() : void {
        // For a free utility plugin, safest default is: do NOT delete data automatically.
        // Uncomment if you want full cleanup.
        // global $wpdb;
        // $wpdb->query( "DROP TABLE IF EXISTS " . self::table_name() );
        // delete_option( 'otterfixer_aibt_bots' );
    }

    private static function table_name() : string {
        global $wpdb;
        return $wpdb->prefix . self::TABLE;
    }

    public static function maybe_log_hit() : void {
        // Only front-end requests.
        if ( is_admin() || wp_doing_ajax() || wp_doing_cron() ) {
            return;
        }

        $ua = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( (string) $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        if ( $ua === '' ) { return; }

        $bots = get_option( 'otterfixer_aibt_bots', array() );
        if ( ! is_array( $bots ) || empty( $bots ) ) { return; }

        $bot_name = '';
        foreach ( $bots as $name => $needle ) {
            $needle = (string) $needle;
            if ( $needle !== '' && stripos( $ua, $needle ) !== false ) {
                $bot_name = (string) $name;
                break;
            }
        }

        if ( $bot_name === '' ) { return; }

        global $wpdb;

        $path = '';
        if ( isset( $_SERVER['REQUEST_URI'] ) ) {
            $path = sanitize_text_field( wp_unslash( (string) $_SERVER['REQUEST_URI'] ) );
        }

        $referer = '';
        if ( isset( $_SERVER['HTTP_REFERER'] ) ) {
            $referer = esc_url_raw( wp_unslash( (string) $_SERVER['HTTP_REFERER'] ) );
        }

        $ip = '';
        if ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
            $raw_ip = sanitize_text_field( wp_unslash( (string) $_SERVER['REMOTE_ADDR'] ) );
            $ip     = self::anonymize_ip( $raw_ip );
        }

        // Store UTC time for consistency.
        $ts = gmdate( 'Y-m-d H:i:s' );

        $wpdb->insert(
            self::table_name(),
            array(
                'ts'      => $ts,
                'bot'     => $bot_name,
                'path'    => $path,
                'referer' => $referer,
                'ip'      => $ip,
                'ua'      => $ua,
            ),
            array( '%s', '%s', '%s', '%s', '%s', '%s' )
        );
    }

    public static function admin_menu() : void {
        // Tools → OtterFixer AI Bot Tracker
        add_management_page(
            __( 'OtterFixer AI Bot Tracker', 'otterfixer-ai-bot-tracker' ),
            __( 'AI Bot Tracker', 'otterfixer-ai-bot-tracker' ),
            self::CAP_VIEW,
            self::SLUG,
            array( __CLASS__, 'render_page' )
        );
    }
    private static function require_admin() : void {
        if ( ! current_user_can( self::CAP_VIEW ) ) {
            wp_die( esc_html__( 'Sorry, you are not allowed to access this page.', 'otterfixer-ai-bot-tracker' ) );
        }
    }

    public static function render_page() : void {
        self::require_admin();

        global $wpdb;
        $table = esc_sql( self::table_name() );

        // Actions
        if ( isset( $_POST['otterfixer_aibt_aibt_action'] ) && $_POST['otterfixer_aibt_aibt_action'] === 'clear' ) {
            if ( ! current_user_can( self::CAP_ADMIN ) ) {
                wp_die( esc_html__( 'You do not have permission to do that.', 'otterfixer-ai-bot-tracker' ) );
            }
            check_admin_referer( 'otterfixer_aibt_aibt_clear' );
            $wpdb->query( "TRUNCATE TABLE {$table}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Logs cleared.', 'otterfixer-ai-bot-tracker' ) . '</p></div>';
        }

        // Summary
        $total = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery

        $since_7  = gmdate( 'Y-m-d H:i:s', time() - 7 * DAY_IN_SECONDS );
        $since_30 = gmdate( 'Y-m-d H:i:s', time() - 30 * DAY_IN_SECONDS );

        $count_7  = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE ts >= %s", $since_7 ) );
        $count_30 = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE ts >= %s", $since_30 ) );

        $by_bot = $wpdb->get_results( "SELECT bot, COUNT(*) AS c FROM {$table} GROUP BY bot ORDER BY c DESC LIMIT 20", ARRAY_A ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery

        // Recent hits
        $recent = $wpdb->get_results( "SELECT ts, bot, path, ip FROM {$table} ORDER BY id DESC LIMIT 200", ARRAY_A ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery

        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'OtterFixer AI Bot Tracker', 'otterfixer-ai-bot-tracker' ); ?></h1>

            <p><?php echo esc_html__( 'This tool logs front-end requests that match common AI/LLM crawler user-agents. Nothing is sent anywhere. Data stays in your database.', 'otterfixer-ai-bot-tracker' ); ?></p>

            <div style="background:#fff;border:1px solid #dcdcde;border-radius:12px;padding:14px 16px;max-width:980px;">
                <h2 style="margin:0 0 10px; font-size:16px;"><?php echo esc_html__( 'How to use', 'otterfixer-ai-bot-tracker' ); ?></h2>
                <ol style="margin:0 0 10px 18px;">
                    <li><?php echo esc_html__( 'Install and activate the plugin.', 'otterfixer-ai-bot-tracker' ); ?></li>
                    <li><?php echo esc_html__( 'Visit this page any time: Tools > AI Bot Tracker.', 'otterfixer-ai-bot-tracker' ); ?></li>
                    <li><?php echo esc_html__( 'Wait for AI crawlers to visit your site. When they do, totals and recent hits will appear below.', 'otterfixer-ai-bot-tracker' ); ?></li>
                </ol>
                <p style="margin:0; opacity:.85;">
                    <?php echo esc_html__( 'Tip: If you have zero hits, that usually just means no matching bot has visited yet. This page only logs front-end requests, not wp-admin.', 'otterfixer-ai-bot-tracker' ); ?>
                </p>
            </div>

            <div style="display:flex; gap:16px; flex-wrap:wrap; margin: 16px 0;">
                <div style="background:#fff;border:1px solid #dcdcde;border-radius:12px;padding:12px 14px;min-width:220px;">
                    <div style="font-size:12px;opacity:.8;"><?php echo esc_html__( 'Total logged hits', 'otterfixer-ai-bot-tracker' ); ?></div>
                    <div style="font-size:22px;font-weight:700;"><?php echo esc_html( number_format_i18n( $total ) ); ?></div>
                </div>
                <div style="background:#fff;border:1px solid #dcdcde;border-radius:12px;padding:12px 14px;min-width:220px;">
                    <div style="font-size:12px;opacity:.8;"><?php echo esc_html__( 'Last 7 days', 'otterfixer-ai-bot-tracker' ); ?></div>
                    <div style="font-size:22px;font-weight:700;"><?php echo esc_html( number_format_i18n( $count_7 ) ); ?></div>
                </div>
                <div style="background:#fff;border:1px solid #dcdcde;border-radius:12px;padding:12px 14px;min-width:220px;">
                    <div style="font-size:12px;opacity:.8;"><?php echo esc_html__( 'Last 30 days', 'otterfixer-ai-bot-tracker' ); ?></div>
                    <div style="font-size:22px;font-weight:700;"><?php echo esc_html( number_format_i18n( $count_30 ) ); ?></div>
                </div>
            </div>

            <h2><?php echo esc_html__( 'Hits by bot', 'otterfixer-ai-bot-tracker' ); ?></h2>
            <?php if ( empty( $by_bot ) ) : ?>
                <p><?php echo esc_html__( 'No bot hits logged yet.', 'otterfixer-ai-bot-tracker' ); ?></p>
            <?php else : ?>
                <table class="widefat striped" style="max-width: 680px;">
                    <thead>
                        <tr>
                            <th><?php echo esc_html__( 'Bot', 'otterfixer-ai-bot-tracker' ); ?></th>
                            <th><?php echo esc_html__( 'Hits', 'otterfixer-ai-bot-tracker' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $by_bot as $row ) : ?>
                            <tr>
                                <td><?php echo esc_html( $row['bot'] ); ?></td>
                                <td><?php echo esc_html( number_format_i18n( (int) $row['c'] ) ); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>

            <h2 style="margin-top:24px;"><?php echo esc_html__( 'Recent hits', 'otterfixer-ai-bot-tracker' ); ?></h2>
            <?php if ( empty( $recent ) ) : ?>
                <p><?php echo esc_html__( 'Nothing logged yet. This page only logs front-end requests, so it will stay empty until a matching bot visits your site.', 'otterfixer-ai-bot-tracker' ); ?></p>
            <?php else : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th style="width:170px;"><?php echo esc_html__( 'UTC time', 'otterfixer-ai-bot-tracker' ); ?></th>
                            <th style="width:160px;"><?php echo esc_html__( 'Bot', 'otterfixer-ai-bot-tracker' ); ?></th>
                            <th><?php echo esc_html__( 'Path', 'otterfixer-ai-bot-tracker' ); ?></th>
                            <th style="width:160px;"><?php echo esc_html__( 'IP', 'otterfixer-ai-bot-tracker' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $recent as $r ) : ?>
                            <tr>
                                <td><?php echo esc_html( $r['ts'] ); ?></td>
                                <td><?php echo esc_html( $r['bot'] ); ?></td>
                                <td><code><?php echo esc_html( $r['path'] ); ?></code></td>
                                <td><?php echo esc_html( $r['ip'] ); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>

            <hr style="margin:24px 0;" />

            <h2><?php echo esc_html__( 'Maintenance', 'otterfixer-ai-bot-tracker' ); ?></h2>
            <form method="post">
                <?php wp_nonce_field( 'otterfixer_aibt_aibt_clear' ); ?>
                <input type="hidden" name="otterfixer_aibt_aibt_action" value="clear" />
                <p>
                    <button type="submit" class="button button-secondary" onclick="return confirm(&quot;Clear all AI bot tracker logs?&quot;);">
                        <?php echo esc_html__( 'Clear logs', 'otterfixer-ai-bot-tracker' ); ?>
                    </button>
                </p>
            </form>

            <p style="margin-top:18px; opacity:.8;">
                <?php echo esc_html__( 'Built in-house at OtterFixer. No ads, no tracking, no bloat.', 'otterfixer-ai-bot-tracker' ); ?>
            </p>
        </div>
        <?php
    }

    /**
     * Anonymise an IP address before storing it.
     *
     * - IPv4: sets the last octet to 0 (e.g. 192.168.1.123 -> 192.168.1.0)
     * - IPv6: keeps the first 4 hextets and zeros the rest (e.g. 2001:db8:abcd:0012:... -> 2001:db8:abcd:12::)
     */
    private static function anonymize_ip( string $ip ) : string {
        $ip = trim( $ip );
        if ( $ip === '' ) {
            return '';
        }

        if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) ) {
            $parts = explode( '.', $ip );
            if ( count( $parts ) === 4 ) {
                $parts[3] = '0';
                return implode( '.', $parts );
            }
            return $ip;
        }

        if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6 ) ) {
            // Normalise then keep first 4 groups.
            $packed = @inet_pton( $ip );
            if ( $packed === false ) {
                return $ip;
            }
            $hex = unpack( 'H*', $packed );
            $hex = is_array( $hex ) && isset( $hex[1] ) ? $hex[1] : '';
            if ( $hex === '' || strlen( $hex ) !== 32 ) {
                return $ip;
            }
            $groups = str_split( $hex, 4 );
            $out    = array_slice( $groups, 0, 4 );
            return strtolower( implode( ':', $out ) ) . '::';
        }

        return $ip;
    }

    private static function help_url() : string {
        return 'https://otterfixer.com/#report-an-issue';
    }

    public static function plugin_action_links( array $links ) : array {
        $report_url = admin_url( 'tools.php?page=' . self::SLUG );
        $report     = '<a href="' . esc_url( $report_url ) . '">' . esc_html__( 'View report', 'otterfixer-ai-bot-tracker' ) . '</a>';
        $help       = '<a href="' . esc_url( self::help_url() ) . '" target="_blank" rel="noopener">' . esc_html__( 'Get help', 'otterfixer-ai-bot-tracker' ) . '</a>';
        array_unshift( $links, $help );
        array_unshift( $links, $report );
        return $links;
    }

    public static function admin_help_notice() : void {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        $screen_id = $screen ? (string) $screen->id : '';

        if ( $screen_id !== 'tools_page_' . self::SLUG ) {
            return;
        }

        if ( get_transient( 'otterfixer_aibt_help_notice_shown' ) ) {
            return;
        }
        set_transient( 'otterfixer_aibt_help_notice_shown', 1, DAY_IN_SECONDS );

        echo '<div class="notice notice-info is-dismissible"><p><strong>OtterFixer:</strong> If something on your site is not working, you can report it and I will take a look. <a href="' . esc_url( self::help_url() ) . '" target="_blank" rel="noopener">Report an issue</a></p></div>';
    }

}


OTTERFIXER_AIBT::init();