<?php
/**
 * Plugin Name: OtterFixer Site Diagnostics
 * Description: Lightweight diagnostics report for WordPress sites. View key site, server, and configuration details in one place and copy a shareable report.
 * Version: 1.0.1
 * Author: OtterFixer (Olivia)
 * License: GPLv2 or later
 * Text Domain: otterfixer-site-diagnostics
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

final class OFX_Site_Diagnostics {

    const VERSION = '1.0.1';
    const CAPABILITY = 'manage_options';
    const SLUG = 'otterfixer-site-diagnostics';

    public static function init() : void {
add_action( 'admin_menu', array( __CLASS__, 'admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_assets' ) );

        add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( __CLASS__, 'plugin_action_links' ) );
        add_action( 'admin_notices', array( __CLASS__, 'admin_help_notice' ) );
    }

    public static function admin_menu() : void {
        add_management_page(
            __( 'OtterFixer Diagnostics', 'otterfixer-site-diagnostics' ),
            __( 'OtterFixer Diagnostics', 'otterfixer-site-diagnostics' ),
            self::CAPABILITY,
            self::SLUG,
            array( __CLASS__, 'render_page' )
        );
    }

    public static function admin_assets( string $hook ) : void {
        if ( $hook !== 'tools_page_' . self::SLUG ) { return; }

        wp_register_style(
            'ofx_site_diagnostics_css',
            plugins_url( 'assets/diagnostics.css', __FILE__ ),
            array(),
            self::VERSION
        );
        wp_register_script(
            'ofx_site_diagnostics_js',
            plugins_url( 'assets/diagnostics.js', __FILE__ ),
            array(),
            self::VERSION,
            true
        );

        wp_enqueue_style( 'ofx_site_diagnostics_css' );
        wp_enqueue_script( 'ofx_site_diagnostics_js' );

        wp_localize_script( 'ofx_site_diagnostics_js', 'OFX_DIAG', array(
            'copiedText' => __( 'Copied', 'otterfixer-site-diagnostics' ),
            'copyText'   => __( 'Copy report', 'otterfixer-site-diagnostics' ),
        ) );
    }

    

    // One place to control the help link
    private static function help_url() : string {
        return 'https://otterfixer.com/#report-an-issue';
    }

    // Add a "Get Help" link on the Plugins page
    public static function plugin_action_links( array $links ) : array {
        $help = '<a href="' . esc_url( self::help_url() ) . '" target="_blank" rel="noopener">Get Help</a>';
        array_unshift( $links, $help );
        return $links;
    }

    // Show a small notice on the Diagnostics screen only
    public static function admin_help_notice() : void {

        if ( ! current_user_can( self::CAPABILITY ) ) {
            return;
        }

        // Only show on this plugin page
        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        $screen_id = $screen ? (string) $screen->id : '';
        if ( $screen_id !== 'tools_page_' . self::SLUG ) {
            return;
        }

        // If multiple OtterFixer plugins add this later, only show one notice total
        if ( get_transient( 'ofx_help_notice_shown' ) ) {
            return;
        }
        set_transient( 'ofx_help_notice_shown', 1, DAY_IN_SECONDS );

        echo '<div class="notice notice-info is-dismissible">';
        echo '<p><strong>OtterFixer:</strong> If something on your site is not working, you can report an issue and I will take a look. ';
        echo '<a href="' . esc_url( self::help_url() ) . '" target="_blank" rel="noopener">Report an issue</a>';
        echo '</p>';
        echo '</div>';
    }

private static function ini_bytes( $val ) : int {
        if ( $val === null || $val === '' ) { return 0; }
        $val = trim( (string) $val );
        $last = strtolower( substr( $val, -1 ) );
        $num = (int) preg_replace( '/[^0-9]/', '', $val );
        switch ( $last ) {
            case 'g': $num *= 1024;
            case 'm': $num *= 1024;
            case 'k': $num *= 1024;
        }
        return $num;
    }

    private static function fmt_bytes( int $bytes ) : string {
        if ( $bytes <= 0 ) { return 'Unknown'; }
        $units = array( 'B','KB','MB','GB','TB' );
        $i = 0;
        while ( $bytes >= 1024 && $i < count( $units ) - 1 ) {
            $bytes /= 1024;
            $i++;
        }
        return ( $i === 0 ) ? (int) $bytes . ' ' . $units[$i] : round( $bytes, 2 ) . ' ' . $units[$i];
    }

    private static function get_active_plugins_list() : array {
        $plugins = (array) get_option( 'active_plugins', array() );
        if ( is_multisite() ) {
            $network = array_keys( (array) get_site_option( 'active_sitewide_plugins', array() ) );
            $plugins = array_unique( array_merge( $plugins, $network ) );
        }
        sort( $plugins );
        return $plugins;
    }

    private static function cache_clues( array $active_plugins ) : array {
        $known = array(
            'wp-rocket/wp-rocket.php' => 'WP Rocket',
            'litespeed-cache/litespeed-cache.php' => 'LiteSpeed Cache',
            'w3-total-cache/w3-total-cache.php' => 'W3 Total Cache',
            'wp-super-cache/wp-cache.php' => 'WP Super Cache',
            'autoptimize/autoptimize.php' => 'Autoptimize',
            'sg-cachepress/sg-cachepress.php' => 'SiteGround Optimiser',
            'redis-cache/redis-cache.php' => 'Redis Object Cache',
            'wphb/wphb.php' => 'Hummingbird',
            'wp-fastest-cache/wpFastestCache.php' => 'WP Fastest Cache',
            'cache-enabler/cache-enabler.php' => 'Cache Enabler',
            'wp-optimize/wp-optimize.php' => 'WP-Optimize',
        );

        $detected = array();
        foreach ( $known as $file => $name ) {
            if ( in_array( $file, $active_plugins, true ) ) { $detected[] = $name; }
        }

        // Drop-in clues
        if ( file_exists( WP_CONTENT_DIR . '/advanced-cache.php' ) ) { $detected[] = 'advanced-cache.php (drop-in)'; }
        if ( file_exists( WP_CONTENT_DIR . '/object-cache.php' ) ) { $detected[] = 'object-cache.php (drop-in)'; }

        return array_values( array_unique( $detected ) );
    }

    private static function rest_api_check() : array {
        $url = rest_url( 'wp/v2/types' );
        $args = array(
            'timeout' => 6,
            'redirection' => 2,
            'headers' => array(
                'Accept' => 'application/json',
            ),
        );
        $res = wp_remote_get( $url, $args );
        if ( is_wp_error( $res ) ) {
            return array( 'status' => 'Fail', 'detail' => $res->get_error_message() );
        }
        $code = wp_remote_retrieve_response_code( $res );
        if ( $code >= 200 && $code < 300 ) {
            return array( 'status' => 'OK', 'detail' => 'HTTP ' . $code );
        }
        return array( 'status' => 'Warn', 'detail' => 'HTTP ' . $code );
    }

    private static function wp_cron_status() : array {
        $disabled = ( defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON );
        $alt = ( defined( 'ALTERNATE_WP_CRON' ) && ALTERNATE_WP_CRON );

        $next = wp_next_scheduled( 'wp_version_check' );
        $next_txt = $next ? gmdate( 'Y-m-d H:i:s', (int) $next ) . ' UTC' : 'None scheduled (or blocked)';

        $status = 'OK';
        if ( $disabled ) { $status = 'Disabled'; }

        return array(
            'status' => $status,
            'disabled' => $disabled ? 'Yes' : 'No',
            'alternate' => $alt ? 'Yes' : 'No',
            'next' => $next_txt,
        );
    }

    private static function get_diag_data() : array {
        global $wpdb;

        $active_plugins = self::get_active_plugins_list();
        $theme = wp_get_theme();

        $memory_limit_wp = defined( 'WP_MEMORY_LIMIT' ) ? (string) WP_MEMORY_LIMIT : 'Not set';
        $memory_limit_php = ini_get( 'memory_limit' );
        $upload_max = ini_get( 'upload_max_filesize' );
        $post_max = ini_get( 'post_max_size' );
        $max_exec = ini_get( 'max_execution_time' );
        $max_input_vars = ini_get( 'max_input_vars' );

        $rest = self::rest_api_check();
        $cron = self::wp_cron_status();

        $data = array(
            'generated' => gmdate( 'Y-m-d H:i:s' ) . ' UTC',
            'site' => array(
                'site_url' => site_url(),
                'home_url' => home_url(),
                'is_https' => is_ssl() ? 'Yes' : 'No',
                'wp_multisite' => is_multisite() ? 'Yes' : 'No',
                'wp_debug' => ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? 'On' : 'Off',
                'wp_environment' => function_exists( 'wp_get_environment_type' ) ? wp_get_environment_type() : 'Unknown',
            ),
            'wordpress' => array(
                'wp_version' => get_bloginfo( 'version' ),
                'language' => get_locale(),
                'timezone' => wp_timezone_string() ? wp_timezone_string() : 'Not set',
                'permalink_structure' => (string) get_option( 'permalink_structure' ),
            ),
            'server' => array(
                'php_version' => PHP_VERSION,
                'server_software' => isset( $_SERVER['SERVER_SOFTWARE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ) : 'Unknown',
                'mysql_version' => method_exists( $wpdb, 'db_version' ) ? $wpdb->db_version() : 'Unknown',
                'memory_limit_wp' => $memory_limit_wp,
                'memory_limit_php' => (string) $memory_limit_php,
                'upload_max_filesize' => (string) $upload_max,
                'post_max_size' => (string) $post_max,
                'max_execution_time' => (string) $max_exec,
                'max_input_vars' => (string) $max_input_vars,
            ),
            'theme' => array(
                'name' => $theme->get( 'Name' ) ?: 'Unknown',
                'version' => $theme->get( 'Version' ) ?: 'Unknown',
                'template' => $theme->get_template(),
                'stylesheet' => $theme->get_stylesheet(),
                'is_child_theme' => is_child_theme() ? 'Yes' : 'No',
            ),
            'plugins' => array(
                'active_count' => (string) count( $active_plugins ),
                'active' => $active_plugins,
                'cache_clues' => self::cache_clues( $active_plugins ),
            ),
            'checks' => array(
                'rest_api' => $rest,
                'wp_cron' => $cron,
            ),
        );

        return $data;
    }

    private static function render_kv_table( array $rows ) : void {
        echo '<table class="widefat striped ofx-kv">';
        echo '<tbody>';
        foreach ( $rows as $k => $v ) {
            echo '<tr>';
            echo '<th scope="row">' . esc_html( $k ) . '</th>';
            echo '<td>' . wp_kses_post( $v ) . '</td>';
            echo '</tr>';
        }
        echo '</tbody>';
        echo '</table>';
    }

    private static function esc_list( array $items ) : string {
        if ( empty( $items ) ) { return '<em>None</em>'; }
        $out = '<ul class="ofx-list">';
        foreach ( $items as $item ) {
            $out .= '<li><code>' . esc_html( $item ) . '</code></li>';
        }
        $out .= '</ul>';
        return $out;
    }

    private static function build_text_report( array $data ) : string {
        $lines = array();

        $lines[] = 'OtterFixer Site Diagnostics';
        $lines[] = 'Generated: ' . $data['generated'];
        $lines[] = '';

        $lines[] = '[Site]';
        foreach ( $data['site'] as $k => $v ) {
            $lines[] = $k . ': ' . $v;
        }
        $lines[] = '';

        $lines[] = '[WordPress]';
        foreach ( $data['wordpress'] as $k => $v ) {
            $lines[] = $k . ': ' . $v;
        }
        $lines[] = '';

        $lines[] = '[Server]';
        foreach ( $data['server'] as $k => $v ) {
            $lines[] = $k . ': ' . $v;
        }
        $lines[] = '';

        $lines[] = '[Theme]';
        foreach ( $data['theme'] as $k => $v ) {
            $lines[] = $k . ': ' . $v;
        }
        $lines[] = '';

        $lines[] = '[Plugins]';
        $lines[] = 'active_count: ' . $data['plugins']['active_count'];
        $lines[] = 'active_plugins:';
        foreach ( $data['plugins']['active'] as $p ) {
            $lines[] = '  - ' . $p;
        }
        $lines[] = 'cache_clues:';
        if ( ! empty( $data['plugins']['cache_clues'] ) ) {
            foreach ( $data['plugins']['cache_clues'] as $c ) {
                $lines[] = '  - ' . $c;
            }
        } else {
            $lines[] = '  - None detected';
        }
        $lines[] = '';

        $lines[] = '[Checks]';
        $lines[] = 'REST API: ' . $data['checks']['rest_api']['status'] . ' (' . $data['checks']['rest_api']['detail'] . ')';
        $lines[] = 'WP-Cron: ' . $data['checks']['wp_cron']['status'] . ' (Disabled: ' . $data['checks']['wp_cron']['disabled'] . ', Alternate: ' . $data['checks']['wp_cron']['alternate'] . ', Next: ' . $data['checks']['wp_cron']['next'] . ')';

        return implode( "\n", $lines );
    }

    public static function render_page() : void {
        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_die( esc_html__( 'You do not have permission to view this page.', 'otterfixer-site-diagnostics' ) );
        }

        $data = self::get_diag_data();
        $report = self::build_text_report( $data );

        $rest_badge = $data['checks']['rest_api']['status'];
        $rest_class = 'ok';
        if ( $rest_badge === 'Warn' ) { $rest_class = 'warn'; }
        if ( $rest_badge === 'Fail' ) { $rest_class = 'fail'; }

        $cron_badge = $data['checks']['wp_cron']['status'];
        $cron_class = ( $cron_badge === 'OK' ) ? 'ok' : 'warn';

        echo '<div class="wrap ofx-wrap">';
        echo '<h1>' . esc_html__( 'OtterFixer Site Diagnostics', 'otterfixer-site-diagnostics' ) . '</h1>';

        echo '<p class="ofx-sub">';
        echo esc_html__( 'A lightweight snapshot of your WordPress setup. Use the copy button to share this report with your developer or support team.', 'otterfixer-site-diagnostics' );
        echo '</p>';

        echo '<div class="ofx-actions">';
        echo '<button type="button" class="button button-primary" id="ofx-copy-btn">' . esc_html__( 'Copy report', 'otterfixer-site-diagnostics' ) . '</button>';
        echo '<span class="ofx-copy-status" id="ofx-copy-status" aria-live="polite"></span>';
        echo '</div>';

        echo '<textarea id="ofx-report" class="ofx-report" readonly>' . esc_textarea( $report ) . '</textarea>';

        echo '<h2>' . esc_html__( 'Quick checks', 'otterfixer-site-diagnostics' ) . '</h2>';
        echo '<div class="ofx-badges">';
        echo '<span class="ofx-badge ' . esc_attr( $rest_class ) . '">REST API: ' . esc_html( $data['checks']['rest_api']['status'] ) . '</span>';
        echo '<span class="ofx-badge ' . esc_attr( $cron_class ) . '">WP-Cron: ' . esc_html( $data['checks']['wp_cron']['status'] ) . '</span>';
        echo '</div>';

        echo '<h2>' . esc_html__( 'Site', 'otterfixer-site-diagnostics' ) . '</h2>';
        self::render_kv_table( array(
            'Site URL' => esc_html( $data['site']['site_url'] ),
            'Home URL' => esc_html( $data['site']['home_url'] ),
            'HTTPS' => esc_html( $data['site']['is_https'] ),
            'Multisite' => esc_html( $data['site']['wp_multisite'] ),
            'WP_DEBUG' => esc_html( $data['site']['wp_debug'] ),
            'Environment' => esc_html( $data['site']['wp_environment'] ),
        ) );

        echo '<h2>' . esc_html__( 'WordPress', 'otterfixer-site-diagnostics' ) . '</h2>';
        self::render_kv_table( array(
            'WordPress version' => esc_html( $data['wordpress']['wp_version'] ),
            'Language' => esc_html( $data['wordpress']['language'] ),
            'Timezone' => esc_html( $data['wordpress']['timezone'] ),
            'Permalink structure' => $data['wordpress']['permalink_structure'] ? '<code>' . esc_html( $data['wordpress']['permalink_structure'] ) . '</code>' : '<em>Default</em>',
        ) );

        echo '<h2>' . esc_html__( 'Server', 'otterfixer-site-diagnostics' ) . '</h2>';
        self::render_kv_table( array(
            'PHP version' => esc_html( $data['server']['php_version'] ),
            'Web server' => esc_html( $data['server']['server_software'] ),
            'MySQL version' => esc_html( $data['server']['mysql_version'] ),
            'WP memory limit' => '<code>' . esc_html( $data['server']['memory_limit_wp'] ) . '</code>',
            'PHP memory limit' => '<code>' . esc_html( $data['server']['memory_limit_php'] ) . '</code>',
            'Upload max filesize' => '<code>' . esc_html( $data['server']['upload_max_filesize'] ) . '</code>',
            'Post max size' => '<code>' . esc_html( $data['server']['post_max_size'] ) . '</code>',
            'Max execution time' => '<code>' . esc_html( $data['server']['max_execution_time'] ) . '</code>',
            'Max input vars' => '<code>' . esc_html( $data['server']['max_input_vars'] ) . '</code>',
        ) );

        echo '<h2>' . esc_html__( 'Theme', 'otterfixer-site-diagnostics' ) . '</h2>';
        self::render_kv_table( array(
            'Theme name' => esc_html( $data['theme']['name'] ),
            'Theme version' => esc_html( $data['theme']['version'] ),
            'Template' => '<code>' . esc_html( $data['theme']['template'] ) . '</code>',
            'Stylesheet' => '<code>' . esc_html( $data['theme']['stylesheet'] ) . '</code>',
            'Child theme' => esc_html( $data['theme']['is_child_theme'] ),
        ) );

        echo '<h2>' . esc_html__( 'Plugins', 'otterfixer-site-diagnostics' ) . '</h2>';
        self::render_kv_table( array(
            'Active plugin count' => esc_html( $data['plugins']['active_count'] ),
            'Cache clues' => ! empty( $data['plugins']['cache_clues'] ) ? esc_html( implode( ', ', $data['plugins']['cache_clues'] ) ) : '<em>None detected</em>',
        ) );

        echo '<h3>' . esc_html__( 'Active plugins', 'otterfixer-site-diagnostics' ) . '</h3>';
        echo self::esc_list( $data['plugins']['active'] );

        echo '<p class="ofx-footer">';
        echo esc_html__( 'Built in-house at OtterFixer by Olivia. Lightweight, no tracking, no ads.', 'otterfixer-site-diagnostics' );
        echo '</p>';

        echo '</div>';
    }
}

OFX_Site_Diagnostics::init();
