<?php
/**
 * Plugin Name: OtterFixer Broken Link Finder
 * Description: Scan your WordPress site for broken internal links and error links (404/403/500) in posts, pages, and menus. View results and export a CSV fix list.
 * Version: 1.0.3
 * Author: OtterFixer
 * License: GPLv2 or later
 * Text Domain: otterfixer-broken-link-finder
 */

if (!defined('ABSPATH')) { exit; }

define('OTTERFIXER_BLF_VERSION', '1.0.2');
define('OTTERFIXER_BLF_TABLE', 'otterfixer_blf_results');

class OtterFixer_Broken_Link_Finder {

    private static $instance = null;

    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));

        add_action('wp_ajax_otterfixer_blf_start_scan', array($this, 'ajax_start_scan'));
        add_action('wp_ajax_otterfixer_blf_process_batch', array($this, 'ajax_process_batch'));

        add_action('admin_post_otterfixer_blf_export_csv', array($this, 'export_csv'));

        add_action('admin_init', array($this, 'maybe_clear_results'));
    }

    public static function activate() {
        self::create_table();
    }

    public static function uninstall() {
        global $wpdb;
        $table = $wpdb->prefix . OTTERFIXER_BLF_TABLE;
        $wpdb->query("DROP TABLE IF EXISTS {$table}");
        delete_transient('otterfixer_blf_queue');
        delete_transient('otterfixer_blf_progress');
    }

    private static function create_table() {
        global $wpdb;
        $table = $wpdb->prefix . OTTERFIXER_BLF_TABLE;

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            link_url TEXT NOT NULL,
            status_code INT(11) NOT NULL DEFAULT 0,
            status_message VARCHAR(190) NOT NULL DEFAULT '',
            source_type VARCHAR(30) NOT NULL DEFAULT '',
            source_id BIGINT(20) UNSIGNED NOT NULL DEFAULT 0,
            source_title VARCHAR(255) NOT NULL DEFAULT '',
            source_url TEXT NOT NULL,
            context VARCHAR(255) NOT NULL DEFAULT '',
            last_checked DATETIME NOT NULL,
            PRIMARY KEY  (id),
            KEY status_code (status_code),
            KEY source_type (source_type),
            KEY source_id (source_id)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    private function current_user_can_manage() {
        return current_user_can('manage_options');
    }

    public function admin_menu() {
        add_management_page(
            __('OtterFixer Broken Link Finder', 'otterfixer-broken-link-finder'),
            __('OtterFixer Broken Link Finder', 'otterfixer-broken-link-finder'),
            'manage_options',
            'otterfixer-broken-link-finder',
            array($this, 'render_admin_page')
        );
    }

    public function enqueue_assets($hook) {
        if ($hook !== 'tools_page_otterfixer-broken-link-finder') {
            return;
        }

        wp_enqueue_style(
            'otterfixer-blf-admin',
            plugins_url('assets/admin.css', __FILE__),
            array(),
            OTTERFIXER_BLF_VERSION
        );

        wp_enqueue_script(
            'otterfixer-blf-admin',
            plugins_url('assets/admin.js', __FILE__),
            array('jquery'),
            OTTERFIXER_BLF_VERSION,
            true
        );

        wp_localize_script('otterfixer-blf-admin', 'OTTERFIXER_BLF', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('otterfixer_blf_nonce'),
        ));
    }

    public function maybe_clear_results() {
        if (!isset($_GET['page']) || $_GET['page'] !== 'otterfixer-broken-link-finder') { return; }
        if (!isset($_GET['otterfixer_blf_clear']) || $_GET['otterfixer_blf_clear'] !== '1') { return; }
        if (!$this->current_user_can_manage()) { return; }

        check_admin_referer('otterfixer_blf_clear_results');
        $this->clear_results();

        wp_safe_redirect(remove_query_arg(array('otterfixer_blf_clear','_wpnonce')));
        exit;
    }

    private function clear_results() {
        global $wpdb;
        $table = $wpdb->prefix . OTTERFIXER_BLF_TABLE;
        $wpdb->query("TRUNCATE TABLE {$table}");
        delete_transient('otterfixer_blf_queue');
        delete_transient('otterfixer_blf_progress');
    }

    public function render_admin_page() {
        if (!$this->current_user_can_manage()) { return; }

        require_once dirname(__FILE__) . '/includes/class-otterfixer-blf-table.php';

        $list_table = new OtterFixer_BLF_List_Table();
        $list_table->prepare_items();

        $export_url = wp_nonce_url(admin_url('admin-post.php?action=otterfixer_blf_export_csv'), 'otterfixer_blf_export_csv');
        $clear_url  = wp_nonce_url(add_query_arg('otterfixer_blf_clear', '1', admin_url('tools.php?page=otterfixer-broken-link-finder')), 'otterfixer_blf_clear_results');

        ?>
        <div class="wrap otterfixer-blf-wrap">
            <h1><?php esc_html_e('OtterFixer Broken Link Finder', 'otterfixer-broken-link-finder'); ?></h1>

            <div class="otterfixer-blf-card">
                <p><?php esc_html_e('Scan your posts, pages, and menus for broken internal links and error links. Results are stored locally in your WordPress database.', 'otterfixer-broken-link-finder'); ?></p>

                <div class="otterfixer-blf-actions">
                    <button id="otterfixer-blf-scan" class="button button-primary"><?php esc_html_e('Scan now', 'otterfixer-broken-link-finder'); ?></button>
                    <a class="button" href="<?php echo esc_url($clear_url); ?>"><?php esc_html_e('Clear results', 'otterfixer-broken-link-finder'); ?></a>
                    <a class="button" href="<?php echo esc_url($export_url); ?>"><?php esc_html_e('Export CSV', 'otterfixer-broken-link-finder'); ?></a>
                </div>

                <div class="otterfixer-blf-progress-wrap" style="display:none;">
                    <div class="otterfixer-blf-progress" aria-hidden="true"><span></span></div>
                    <div class="otterfixer-blf-progress-text"></div>
                </div>

                <details class="otterfixer-blf-help">
                    <summary><?php esc_html_e('What counts as a broken link?', 'otterfixer-broken-link-finder'); ?></summary>
                    <ul>
                        <li><?php esc_html_e('404 / 410: missing page (classic broken link).', 'otterfixer-broken-link-finder'); ?></li>
                        <li><?php esc_html_e('403: blocked access.', 'otterfixer-broken-link-finder'); ?></li>
                        <li><?php esc_html_e('500 / 502 / 503: server error (usually needs a developer or hosting check).', 'otterfixer-broken-link-finder'); ?></li>
                        <li><?php esc_html_e('301 / 302: redirect (not broken, but you may want to update the link).', 'otterfixer-broken-link-finder'); ?></li>
                    </ul>
                </details>
            </div>

            <hr />

            <form method="get">
                <input type="hidden" name="page" value="otterfixer-broken-link-finder" />
                <?php $list_table->display(); ?>
            </form>
        </div>
        <?php
    }

    public function ajax_start_scan() {
        if (!$this->current_user_can_manage()) { wp_send_json_error(array('message' => 'Forbidden')); }
        check_ajax_referer('otterfixer_blf_nonce', 'nonce');

        // Clear previous results
        $this->clear_results();

        $items = $this->build_scan_queue();

        set_transient('otterfixer_blf_queue', $items, HOUR_IN_SECONDS);
        set_transient('otterfixer_blf_progress', array(
            'total' => count($items),
            'done'  => 0,
        ), HOUR_IN_SECONDS);

        wp_send_json_success(array(
            'total' => count($items),
        ));
    }

    public function ajax_process_batch() {
        if (!$this->current_user_can_manage()) { wp_send_json_error(array('message' => 'Forbidden')); }
        check_ajax_referer('otterfixer_blf_nonce', 'nonce');

        $batch_size = isset($_POST['batchSize']) ? max(1, min(50, intval($_POST['batchSize']))) : 20;

        $queue = get_transient('otterfixer_blf_queue');
        $progress = get_transient('otterfixer_blf_progress');

        if (!is_array($queue) || !is_array($progress)) {
            wp_send_json_error(array('message' => __('No scan queue found. Please click "Scan now" again.', 'otterfixer-broken-link-finder')));
        }

        $total = intval($progress['total']);
        $done  = intval($progress['done']);

        $batch = array_splice($queue, 0, $batch_size);

        foreach ($batch as $item) {
            $result = $this->check_url($item['link_url']);
            $this->store_result(array_merge($item, $result));
            $done++;
        }

        set_transient('otterfixer_blf_queue', $queue, HOUR_IN_SECONDS);
        set_transient('otterfixer_blf_progress', array('total' => $total, 'done' => $done), HOUR_IN_SECONDS);

        $finished = ($done >= $total) || empty($queue);

        if ($finished) {
            delete_transient('otterfixer_blf_queue');
        }

        wp_send_json_success(array(
            'total'    => $total,
            'done'     => $done,
            'finished' => $finished,
        ));
    }

    private function store_result($data) {
        global $wpdb;
        $table = $wpdb->prefix . OTTERFIXER_BLF_TABLE;

        $wpdb->insert($table, array(
            'link_url'       => $data['link_url'],
            'status_code'    => intval($data['status_code']),
            'status_message' => substr(sanitize_text_field($data['status_message']), 0, 190),
            'source_type'    => sanitize_text_field($data['source_type']),
            'source_id'      => intval($data['source_id']),
            'source_title'   => substr(sanitize_text_field($data['source_title']), 0, 255),
            'source_url'     => $data['source_url'],
            'context'        => substr(sanitize_text_field($data['context']), 0, 255),
            'last_checked'   => current_time('mysql'),
        ), array('%s','%d','%s','%s','%d','%s','%s','%s','%s'));
    }

    private function check_url($url) {
        $args = array(
            'timeout'     => 12,
            'redirection' => 0,
            'headers'     => array('User-Agent' => 'OtterFixer Broken Link Finder/' . OTTERFIXER_BLF_VERSION),
        );

        $response = wp_remote_head($url, $args);

        // Some servers don't support HEAD
        if (is_wp_error($response) || (isset($response['response']['code']) && intval($response['response']['code']) === 405)) {
            $response = wp_remote_get($url, $args);
        }

        if (is_wp_error($response)) {
            return array(
                'status_code'    => 0,
                'status_message' => $response->get_error_message(),
            );
        }

        $code = isset($response['response']['code']) ? intval($response['response']['code']) : 0;
        $msg  = isset($response['response']['message']) ? $response['response']['message'] : '';

        return array(
            'status_code'    => $code,
            'status_message' => $msg,
        );
    }

    private function build_scan_queue() {
        $queue = array();
        $seen  = array();

        $home = home_url('/');
        $home_host = wp_parse_url($home, PHP_URL_HOST);

        // Posts + pages
        $post_ids = get_posts(array(
            'post_type'      => array('post', 'page'),
            'post_status'    => 'any',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ));

        foreach ($post_ids as $pid) {
            $post = get_post($pid);
            if (!$post) { continue; }

            $source_title = get_the_title($pid);
            $source_url   = get_permalink($pid);

            $links = $this->extract_links_from_html($post->post_content);

            foreach ($links as $link) {
                $norm = $this->normalise_internal_url($link, $home, $home_host);
                if (!$norm) { continue; }

                $key = $pid . '|post|' . $norm;
                if (isset($seen[$key])) { continue; }
                $seen[$key] = 1;

                $queue[] = array(
                    'link_url'     => $norm,
                    'source_type'  => 'post',
                    'source_id'    => $pid,
                    'source_title' => $source_title,
                    'source_url'   => $source_url,
                    'context'      => 'content',
                );
            }
        }

        // Menus
        $menus = wp_get_nav_menus();
        foreach ($menus as $menu) {
            $items = wp_get_nav_menu_items($menu->term_id);
            if (empty($items)) { continue; }

            foreach ($items as $mi) {
                $raw = isset($mi->url) ? $mi->url : '';
                $norm = $this->normalise_internal_url($raw, $home, $home_host);
                if (!$norm) { continue; }

                $key = $menu->term_id . '|menu|' . $norm;
                if (isset($seen[$key])) { continue; }
                $seen[$key] = 1;

                $queue[] = array(
                    'link_url'     => $norm,
                    'source_type'  => 'menu',
                    'source_id'    => intval($menu->term_id),
                    'source_title' => $menu->name,
                    'source_url'   => admin_url('nav-menus.php'),
                    'context'      => 'menu: ' . (isset($mi->title) ? $mi->title : ''),
                );
            }
        }

        return $queue;
    }

    private function extract_links_from_html($html) {
        if (!is_string($html) || $html === '') { return array(); }

        $links = array();

        // Fast regex extraction for href
        if (preg_match_all('/href\s*=\s*([\"\'])(.*?)\1/i', $html, $matches)) {
            foreach ($matches[2] as $u) {
                $u = trim(html_entity_decode($u, ENT_QUOTES));
                if ($u !== '') {
                    $links[] = $u;
                }
            }
        }

        return array_values(array_unique($links));
    }

    private function normalise_internal_url($url, $home, $home_host) {
        $url = trim((string)$url);
        if ($url === '') { return ''; }

        // Ignore anchors, mailto, tel, javascript
        if (strpos($url, '#') === 0) { return ''; }
        if (preg_match('/^(mailto:|tel:|javascript:)/i', $url)) { return ''; }

        // Make absolute for relative URLs
        if (strpos($url, '//') === 0) {
            $url = (is_ssl() ? 'https:' : 'http:') . $url;
        } elseif (strpos($url, 'http://') !== 0 && strpos($url, 'https://') !== 0) {
            $url = home_url($url);
        }

        $host = wp_parse_url($url, PHP_URL_HOST);
        if (!$host || strtolower($host) !== strtolower($home_host)) {
            return '';
        }

        // Strip fragments
        $url = preg_replace('/#.*$/', '', $url);

        return esc_url_raw($url);
    }

    public function export_csv() {
        if (!$this->current_user_can_manage()) { wp_die('Forbidden'); }
        check_admin_referer('otterfixer_blf_export_csv');

        global $wpdb;
        $table = $wpdb->prefix . OTTERFIXER_BLF_TABLE;

        $rows = $wpdb->get_results("SELECT * FROM {$table} ORDER BY last_checked DESC", ARRAY_A);

        nocache_headers();
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=otterfixer-broken-link-finder-results.csv');

        $out = fopen('php://output', 'w');

        fputcsv($out, array('Status', 'Link URL', 'Found on', 'Source type', 'Context', 'Last checked'));

        foreach ($rows as $r) {
            fputcsv($out, array(
                $r['status_code'],
                $r['link_url'],
                $r['source_title'] . ' - ' . $r['source_url'],
                $r['source_type'],
                $r['context'],
                $r['last_checked'],
            ));
        }

        fclose($out);
        exit;
    }
}

register_activation_hook(__FILE__, array('OtterFixer_Broken_Link_Finder', 'activate'));
register_uninstall_hook(__FILE__, array('OtterFixer_Broken_Link_Finder', 'uninstall'));

OtterFixer_Broken_Link_Finder::instance();


// --- OtterFixer Get Help integration ---
add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), function ( $links ) {
    $help = '<a href="https://otterfixer.com/#report-an-issue" target="_blank" rel="noopener">Get Help</a>';
    array_unshift( $links, $help );
    return $links;
});

add_action( 'admin_notices', function () {

    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
    if ( ! $screen || strpos( $screen->id, 'otterfixer-broken-link-finder' ) === false ) {
        return;
    }

    if ( get_transient( 'ofx_help_notice_shown' ) ) {
        return;
    }
    set_transient( 'ofx_help_notice_shown', 1, DAY_IN_SECONDS );

    echo '<div class="notice notice-info is-dismissible"><p><strong>OtterFixer:</strong> Found broken links or issues you are unsure how to fix? <a href="https://otterfixer.com/#report-an-issue" target="_blank" rel="noopener">Report an issue</a>.</p></div>';
});
