<?php
if (!defined('ABSPATH')) { exit; }

if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class OtterFixer_BLF_List_Table extends WP_List_Table {

    public function __construct() {
        parent::__construct(array(
            'singular' => 'link',
            'plural'   => 'links',
            'ajax'     => false,
        ));
    }

    public function get_columns() {
        return array(
            'status_code'  => __('Status', 'otterfixer-broken-link-finder'),
            'link_url'     => __('Link URL', 'otterfixer-broken-link-finder'),
            'source'       => __('Found on', 'otterfixer-broken-link-finder'),
            'context'      => __('Context', 'otterfixer-broken-link-finder'),
            'last_checked' => __('Last checked', 'otterfixer-broken-link-finder'),
        );
    }

    protected function get_sortable_columns() {
        return array(
            'status_code'  => array('status_code', false),
            'last_checked' => array('last_checked', true),
        );
    }

    public function column_status_code($item) {
        $code = intval($item['status_code']);
        $label = $code ? (string)$code : __('Error', 'otterfixer-broken-link-finder');

        $cls = 'otterfixer-badge-ok';
        if ($code >= 400 || $code === 0) { $cls = 'otterfixer-badge-bad'; }
        elseif ($code >= 300) { $cls = 'otterfixer-badge-warn'; }

        $msg = trim((string)($item['status_message'] ?? ''));
        $msg_html = $msg ? '<div class="otterfixer-muted">'. esc_html($msg) .'</div>' : '';

        return '<span class="otterfixer-badge '. esc_attr($cls) .'">'. esc_html($label) .'</span>' . $msg_html;
    }

    public function column_link_url($item) {
        $u = (string)$item['link_url'];
        return '<a href="'. esc_url($u) .'" target="_blank" rel="noopener">'. esc_html($u) .'</a>';
    }

    public function column_source($item) {
        $title = (string)($item['source_title'] ?? '');
        $url   = (string)($item['source_url'] ?? '');

        $line1 = $title ? esc_html($title) : __('(Unknown)', 'otterfixer-broken-link-finder');
        $line2 = $url ? '<div class="otterfixer-muted"><a href="'. esc_url($url) .'" target="_blank" rel="noopener">'. esc_html($url) .'</a></div>' : '';

        return $line1 . $line2;
    }

    public function column_context($item) {
        $ctx = (string)($item['context'] ?? '');
        return $ctx ? esc_html($ctx) : '&#8212;';
    }

    public function column_last_checked($item) {
        return esc_html((string)($item['last_checked'] ?? ''));
    }

    public function column_default($item, $column_name) {
        if (isset($item[$column_name])) {
            return esc_html((string)$item[$column_name]);
        }
        return '';
    }

    public function prepare_items() {
        global $wpdb;
        $table = $wpdb->prefix . OTTERFIXER_BLF_TABLE;

        $per_page = 20;
        $paged = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;

        $orderby = isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'last_checked';
        $order   = isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'DESC';

        $allowed_orderby = array('status_code', 'last_checked');
        if (!in_array($orderby, $allowed_orderby, true)) { $orderby = 'last_checked'; }
        $order = strtoupper($order) === 'ASC' ? 'ASC' : 'DESC';

        // Filter
        $filter = isset($_GET['status_filter']) ? sanitize_text_field($_GET['status_filter']) : 'all';
        $where = '1=1';
        if ($filter === 'broken') {
            $where = '(status_code >= 400 OR status_code = 0)';
        } elseif ($filter === 'redirects') {
            $where = '(status_code >= 300 AND status_code < 400)';
        } elseif ($filter === 'ok') {
            $where = '(status_code >= 200 AND status_code < 300)';
        }

        $total_items = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE {$where}");

        $offset = ($paged - 1) * $per_page;
        $items = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE {$where} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        $columns  = $this->get_columns();
        $hidden   = array();
        $sortable = $this->get_sortable_columns();
        $primary  = 'link_url';
        $this->_column_headers = array($columns, $hidden, $sortable, $primary);

        $this->items = is_array($items) ? $items : array();

        $this->set_pagination_args(array(
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page),
        ));
    }

    protected function extra_tablenav($which) {
        if ($which !== 'top') { return; }
        $current = isset($_GET['status_filter']) ? sanitize_text_field($_GET['status_filter']) : 'all';

        echo '<div class="alignleft actions">';
        echo '<label class="screen-reader-text" for="status_filter">'. esc_html__('Filter by status', 'otterfixer-broken-link-finder') .'</label>';
        echo '<select name="status_filter" id="status_filter">';
        echo '<option value="all" '. selected($current, 'all', false) .'>'. esc_html__('All', 'otterfixer-broken-link-finder') .'</option>';
        echo '<option value="broken" '. selected($current, 'broken', false) .'>'. esc_html__('Broken / Errors (4xx/5xx)', 'otterfixer-broken-link-finder') .'</option>';
        echo '<option value="redirects" '. selected($current, 'redirects', false) .'>'. esc_html__('Redirects (3xx)', 'otterfixer-broken-link-finder') .'</option>';
        echo '<option value="ok" '. selected($current, 'ok', false) .'>'. esc_html__('OK (2xx)', 'otterfixer-broken-link-finder') .'</option>';
        echo '</select>';

        submit_button(__('Filter', 'otterfixer-broken-link-finder'), '', 'filter_action', false);
        echo '</div>';
    }
}
